/******************************************************************************
 *  swkey.cpp - code for base class 'SWKey'.  SWKey is the basis for all
 *	        types of keys for indexing into modules (e.g. verse, word,
 *		place, etc.)
 */

#include <swkey.h>
#include <utilfuns.h>
#include <string.h>

/******************************************************************************
 * SWKey Constructor - initializes instance of SWKey
 *
 * ENT:	ikey - text key
 */

SWKey::SWKey(const char *ikey)
{
	persist = 0;
	keytext = 0;
	error   = 0;
	stdstr(&keytext, ikey);
}


/******************************************************************************
 * SWKey Destructor - cleans up instance of SWKey
 */

SWKey::~SWKey()
{
	if (keytext)
		delete [] keytext;
}


/******************************************************************************
 * SWKey::Persist - Gets whether this object itself persists within a
 *			module that it was used to SetKey or just a copy.
 *			(1 - persists in module; 0 - a copy is attempted
 *
 * RET:	value of persist
 */

char SWKey::Persist() const
{
	return persist;
}


/******************************************************************************
 * SWKey::Persist - Set/gets whether this object itself persists within a
 *			module that it was used to SetKey or just a copy.
 *			(1 - persists in module; 0 - a copy is attempted
 *
 * ENT:	ipersist - value which to set persist
 *		[-1] - only get
 *
 * RET:	value of persist
 */

char SWKey::Persist(char ipersist)
{
	if (ipersist != -1)
		persist = ipersist;

	return persist;
}


/******************************************************************************
 * SWKey::Error - Gets and clears error status
 *
 * RET:	error status
 */

char SWKey::Error()
{
	char retval = error;

	error = 0;
	return retval;
}


/******************************************************************************
 * SWKey::operator = Equates this SWKey to a character string
 *
 * ENT:	ikey - other swkey object
 */

SWKey &SWKey::operator =(const char *ikey)
{
	stdstr(&keytext, ikey);

	return *this;
}


/******************************************************************************
 * SWKey::operator = Equates this SWKey to another SWKey object
 *
 * ENT:	ikey - other swkey object
 */

SWKey &SWKey::operator =(SWKey &ikey)
{
// not desirable	Persist(ikey.Persist());
	stdstr(&keytext, (const char *)ikey);

	return *this;
}


/******************************************************************************
 * SWKey::operator char * - returns text key if (char *) cast is requested
 */

SWKey::operator const char *()
{
	return keytext;
}


/******************************************************************************
 * SWKey::operator == Compares another VerseKey object
 *
 * ENT:	ikey - key to compare with this one
 *
 * RET:	if this key is equal to ikey
 */

char SWKey::operator ==(SWKey &ikey)
{
	return !strcmp(*this, (const char *)ikey);
}


/******************************************************************************
 * SWKey::operator > Compares another VerseKey object
 *
 * ENT:	ikey - key to compare with this one
 *
 * RET:	if this key is greater than ikey
 */

char SWKey::operator >(SWKey &ikey)
{
	return (strcmp(*this, ikey) > 0);
}


/******************************************************************************
 * SWKey::operator < Compares another VerseKey object
 *
 * ENT:	ikey - key to compare with this one
 *
 * RET:	if this key is less than ikey
 */

char SWKey::operator <(SWKey &ikey)
{
	return (strcmp(*this, ikey) < 0);
}


/******************************************************************************
 * SWKey::operator =(POSITION)	- Positions this key if applicable
 */

SWKey &SWKey::operator =(POSITION)
{
	return *this;
}


/******************************************************************************
 * SWKey::operator +=	- Increments key a number of entries
 *
 * ENT:	increment	- Number of entries to jump forward
 *
 * RET: *this
 */

SWKey &SWKey::operator += (int)
{
	error = KEYERR_OUTOFBOUNDS;
	return *this;
}


/******************************************************************************
 * SWKey::operator -=	- Decrements key a number of entries
 *
 * ENT:	decrement	- Number of entries to jump backward
 *
 * RET: *this
 */

SWKey &SWKey::operator -= (int)
{
	error = KEYERR_OUTOFBOUNDS;
	return *this;
}
