
{*******************************************************}
{    The Delphi Unicode Controls Project                }
{                                                       }
{      http://home.ccci.org/wolbrink                    }
{                                                       }
{ Copyright (c) 2002, Troy Wolbrink (wolbrink@ccci.org) }
{                                                       }
{*******************************************************}

unit TntClasses;

{ If you want to use JCLUnicode that comes with Jedi Component Library,
    define JCL as a "Conditional Define" in the project options. }

interface

{$IFNDEF VER130}
{$WARN SYMBOL_PLATFORM OFF} { We are going to use Win32 specific symbols! }
{$ENDIF}

uses Classes, SysUtils, Windows, ActiveX, {$IFDEF JCL} JclUnicode, {$ELSE} Unicode, {$ENDIF}
  Controls, Graphics;

{TNT-WARN Char}
{TNT-WARN PChar}
{TNT-WARN String}

{TNT-WARN LPSTR}
{TNT-WARN PLPSTR}
{TNT-WARN LPCSTR}
{TNT-WARN LPCTSTR}
{TNT-WARN LPTSTR}

// SBCS and MBCS functions with WideString replacements in Unicode.pas
{TNT-WARN QuotedStr}                    {TNT-WARN AnsiQuotedStr}
{TNT-WARN StrComp}                      {TNT-WARN AnsiStrComp}
{TNT-WARN StrIComp}                     {TNT-WARN AnsiStrIComp}
{TNT-WARN StrLComp}                     {TNT-WARN AnsiStrLComp}
{TNT-WARN StrLIComp}                    {TNT-WARN AnsiStrLIComp}
{TNT-WARN StrLower}                     {TNT-WARN AnsiStrLower}
{TNT-WARN StrPos}                       {TNT-WARN AnsiStrPos}
{TNT-WARN StrRScan}                     {TNT-WARN AnsiStrRScan}
{TNT-WARN StrScan}                      {TNT-WARN AnsiStrScan}
{TNT-WARN StrUpper}                     {TNT-WARN AnsiStrUpper}
                                        {TNT-WARN AnsiExtractQuotedStr}

{TNT-WARN AnsiPos}     // Pos works with WideString fine
{TNT-WARN StrPCopyW}   // accepts ansi string parameter
{TNT-WARN StrPLCopyW}  // accepts ansi string parameter

(* MBCS Byte Type Procs *)
{TNT-WARN ByteType}
{TNT-WARN StrByteType}
{TNT-WARN ByteToCharIndex}
{TNT-WARN ByteToCharLen}
{TNT-WARN CharToByteIndex}
{TNT-WARN CharToByteLen}

{TNT-WARN VarToStr}
{TNT-WARN VarToStrDef}
{TNT-WARN FmtStr}
{TNT-WARN Format}
{TNT-WARN FormatBuf}

{TNT-WARN TStringStream}   // TO DO: Implement a TWideStringStream

function MakeObjectInstance(Method: TWndMethod): Pointer;
procedure FreeObjectInstance(ObjectInstance: Pointer);

const
  CR = WideChar($D);
  LF = WideChar($A);
  CRLF = WideString(#$D#$A);
  LineSeparator = WideChar($2028);

// Tnt Original Procs

function WinCheckH(RetVal: Cardinal): Cardinal;
function WinCheckFileH(RetVal: Cardinal): Cardinal;
function WinCheckP(RetVal: Pointer): Pointer;

function FontCharSetToCodePage(FontCharSet: TFontCharSet): Cardinal;
{$IFDEF JCL}
function CodePageToWideString(const A: AnsiString; CodePage: Cardinal; dwFlags: Cardinal = 0): WideString;
{$ENDIF}
function WideStringToCodePage(const W: WideString; CodePage: Cardinal; dwFlags: Cardinal = 0): AnsiString;
function UCS2ToWideString(const Value: AnsiString): WideString;
function WideStringToUCS2(const Value: WideString): AnsiString;

function WideTextPos(const SubStr, S: WideString): Integer;
function IsWideCharDigit(WC: WideChar): Boolean;
function IsWideCharAlpha(WC: WideChar): Boolean;
function IsWideCharAlphaNumeric(WC: WideChar): Boolean;

function WideGetModuleFileName(Instance: HModule): WideString;
function ClassIsRegistered(const clsid: TCLSID): Boolean;

// Tnt-System

{TNT-WARN ParamCount}
function WideParamCount: Integer;
{TNT-WARN ParamStr}
function WideParamStr(Index: Integer): WideString;

// Tnt-Windows

// -- the Unicode form of these functions are supported on Windows 95/98/ME --
//   (Even though EnumResourceTypes, EnumResourceNames and EnumResourceLanguages are supposed
//      to support the Unicode form, they have caused access violations in testing on Win95.)
{TNT-WARN ExtTextOut}
{TNT-WARN ExtTextOutA}
{TNT-WARN FindResource}
{TNT-WARN FindResourceA}
{TNT-WARN FindResourceEx}
{TNT-WARN FindResourceExA}
{TNT-WARN GetCharWidth}
{TNT-WARN GetCharWidthA}
{TNT-WARN GetCommandLine}
{TNT-WARN GetCommandLineA}
{TNT-WARN GetTextExtentPoint}
{TNT-WARN GetTextExtentPointA}
{TNT-WARN GetTextExtentPoint32}
{TNT-WARN GetTextExtentPoint32A}
{TNT-WARN lstrcat}
{TNT-WARN lstrcatA}
{TNT-WARN lstrcpy}
{TNT-WARN lstrcpyA}
{TNT-WARN lstrlen}
{TNT-WARN lstrlenA}
{TNT-WARN MessageBox}
{TNT-WARN MessageBoxA}
{TNT-WARN MessageBoxEx}
{TNT-WARN MessageBoxExA}
{TNT-WARN TextOut}
{TNT-WARN TextOutA}

{TNT-WARN CreateFile}
{TNT-WARN CreateFileA}
function Tnt_CreateFileW(lpFileName: PWideChar; dwDesiredAccess, dwShareMode: DWORD;
  lpSecurityAttributes: PSecurityAttributes; dwCreationDisposition, dwFlagsAndAttributes: DWORD;
    hTemplateFile: THandle): THandle;
function Tnt_Is_IntResource(ResStr: LPCWSTR): Boolean;
{TNT-WARN FindFirstFile}
{TNT-WARN FindFirstFileA}
function Tnt_FindFirstFileW(lpFileName: PWideChar; var lpFindFileData: TWIN32FindDataW): THandle;
{TNT-WARN FindNextFile}
{TNT-WARN FindNextFileA}
function Tnt_FindNextFileW(hFindFile: THandle; var lpFindFileData: TWIN32FindDataW): BOOL;
{TNT-WARN DrawText}
{TNT-WARN DrawTextA}
function Tnt_DrawTextW(hDC: HDC; lpString: PWideChar; nCount: Integer;
  var lpRect: TRect; uFormat: UINT): Integer;
{TNT-WARN GetDiskFreeSpace}
{TNT-WARN GetDiskFreeSpaceA}
function Tnt_GetDiskFreeSpaceW(lpRootPathName: PWideChar; var lpSectorsPerCluster,
  lpBytesPerSector, lpNumberOfFreeClusters, lpTotalNumberOfClusters: DWORD): BOOL;
{TNT-WARN ShellExecute}
{TNT-WARN ShellExecuteA}
function Tnt_ShellExecuteW(hWnd: HWND; Operation, FileName, Parameters,
  Directory: PWideChar; ShowCmd: Integer): HINST;
{TNT-WARN LoadLibraryEx}
{TNT-WARN LoadLibraryExA}
function Tnt_LoadLibraryExW(lpLibFileName: PWideChar; hFile: THandle; dwFlags: DWORD): HMODULE;
{TNT-WARN CreateProcess}
{TNT-WARN CreateProcessA}
function Tnt_CreateProcessW(lpApplicationName: PWideChar; lpCommandLine: PWideChar;
  lpProcessAttributes, lpThreadAttributes: PSecurityAttributes;
    bInheritHandles: BOOL; dwCreationFlags: DWORD; lpEnvironment: Pointer;
      lpCurrentDirectory: PWideChar; const lpStartupInfo: TStartupInfo;
        var lpProcessInformation: TProcessInformation): BOOL;

// Tnt-SysUtils
var Win32PlatformIsUnicode: Boolean;
var Win32PlatformIsXP: Boolean;

{$IFDEF VER130}
const sLineBreak = #13#10;
const PathDelim = '\';
const DriveDelim = ':';
const PathSep = ';';

procedure RaiseLastOSError;
function WideFormat(const FormatStr: WideString; const Args: array of const): WideString;
{TNT-WARN CompareStr}
{TNT-WARN AnsiCompareStr}
function WideCompareStr(const W1, W2: WideString): Integer;
{TNT-WARN AnsiSameStr}
function WideSameStr(const W1, W2: WideString): Boolean;
{TNT-WARN SameText}
{TNT-WARN AnsiSameText}
function WideSameText(const W1, W2: WideString): Boolean;
{$ENDIF}
{TNT-WARN CompareText}
{TNT-WARN AnsiCompareText}
function WideCompareText(const W1, W2: WideString): Integer;

{TNT-WARN UpperCase}
{TNT-WARN AnsiUpperCase}
{TNT-WARN WideUpperCase}
function Unicode_WideUpperCase(const S: WideString): WideString;
{TNT-WARN LowerCase}
{TNT-WARN AnsiLowerCase}
{TNT-WARN WideLowerCase}
function Unicode_WideLowerCase(const S: WideString): WideString;
{TNT-WARN AnsiLastChar}
{TNT-WARN AnsiStrLastChar}
function WideLastChar(W: WideString): WideChar;
{TNT-WARN StringReplace}
function WideStringReplace(const S, OldPattern, NewPattern: WideString;
  Flags: TReplaceFlags; WholeWord: Boolean = False): WideString;
{TNT-WARN AdjustLineBreaks}
type TTntTextLineBreakStyle = (tlbsLF, tlbsCRLF, tlbsCR);
function TntAdjustLineBreaksLength(const S: WideString; Style: TTntTextLineBreakStyle = tlbsCRLF): Integer;
function TntAdjustLineBreaks(const S: WideString; Style: TTntTextLineBreakStyle = tlbsCRLF): WideString;
{TNT-WARN WrapText}
function WideWrapText(const Line, BreakStr: WideString; const BreakChars: TSysCharSet;
  MaxCol: Integer): WideString; overload;
function WideWrapText(const Line: WideString; MaxCol: Integer): WideString; overload;

{TNT-WARN FileCreate}
function WideFileCreate(const FileName: WideString): Integer;
{TNT-WARN FileOpen}
function WideFileOpen(const FileName: WideString; Mode: LongWord): Integer;

function ValidDateTimeStr(Str: WideString): Boolean;
function ValidDateStr(Str: WideString): Boolean;
function ValidTimeStr(Str: WideString): Boolean;

{TNT-WARN StrToDateTime}
function TntStrToDateTime(Str: WideString): TDateTime;
{TNT-WARN StrToDate}
function TntStrToDate(Str: WideString): TDateTime;
{TNT-WARN StrToTime}
function TntStrToTime(Str: WideString): TTime;

// FindFile - warning on TSearchRec is all that is necessary.
{TNT-WARN TSearchRec}
type
  TSearchRecW = record
    Time: Integer;
    Size: Integer;
    Attr: Integer;
    Name: WideString;
    ExcludeAttr: Integer;
    FindHandle: THandle;
    FindData: TWin32FindDataW;
  end;
function WideFindFirst(const Path: WideString; Attr: Integer; var F: TSearchRecW): Integer;
function WideFindNext(var F: TSearchRecW): Integer;
procedure WideFindClose(var F: TSearchRecW);

{TNT-WARN DirectoryExists}
function WideDirectoryExists(const Name: WideString): Boolean;
{TNT-WARN FileExists}
function WideFileExists(const Name: WideString): Boolean;
{TNT-WARN FileGetAttr}
function WideFileGetAttr(const FileName: WideString): Cardinal;
{TNT-WARN FileSetAttr}
function WideFileSetAttr(const FileName: WideString; Attr: Integer): Boolean;

{TNT-WARN SameFileName}           // doesn't apply to Unicode filenames, use WideSameText
{TNT-WARN AnsiCompareFileName}    // doesn't apply to Unicode filenames, use WideCompareText
{TNT-WARN AnsiLowerCaseFileName}  // doesn't apply to Unicode filenames, use WideLowerCase
{TNT-WARN AnsiUpperCaseFileName}  // doesn't apply to Unicode filenames, use WideUpperCase

{TNT-WARN IncludeTrailingBackslash}
function WideIncludeTrailingBackslash(const S: WideString): WideString;
{TNT-WARN ExcludeTrailingBackslash}
function WideExcludeTrailingBackslash(const S: WideString): WideString;
{TNT-WARN IsDelimiter}
function WideIsDelimiter(const Delimiters, S: WideString; Index: Integer): Boolean;
{TNT-WARN IsPathDelimiter}
function WideIsPathDelimiter(const S: WideString; Index: Integer): Boolean;
{TNT-WARN LastDelimiter}
function WideLastDelimiter(const Delimiters, S: WideString): Integer;
{TNT-WARN ChangeFileExt}
function WideChangeFileExt(const FileName, Extension: WideString): WideString;
{TNT-WARN ExtractFilePath}
function WideExtractFilePath(const FileName: WideString): WideString;
{TNT-WARN ExtractFileDir}
function WideExtractFileDir(const FileName: WideString): WideString;
{TNT-WARN ExtractFileDrive}
function WideExtractFileDrive(const FileName: WideString): WideString;
{TNT-WARN ExtractFileName}
function WideExtractFileName(const FileName: WideString): WideString;
{TNT-WARN ExtractFileExt}
function WideExtractFileExt(const FileName: WideString): WideString;
{TNT-WARN ForceDirectories}
function WideForceDirectories(Dir: WideString): Boolean;
{TNT-WARN FileSearch}
function WideFileSearch(const Name, DirList: WideString): WideString;
{TNT-WARN ExtractRelativePath}
function WideExtractRelativePath(const BaseName, DestName: WideString): WideString;

{TNT-WARN RenameFile}
function WideRenameFile(const OldName, NewName: WideString): Boolean;
{TNT-WARN DeleteFile}
function WideDeleteFile(const FileName: WideString): Boolean;
{TNT-WARN CopyFile}
function WideCopyFile(FromFile, ToFile: WideString; FailIfExists: Boolean): Boolean;


type
{TNT-WARN TFileStream}
  TTntFileStream = class(THandleStream)
  public
    constructor Create(const FileName: WideString; Mode: Word);
    destructor Destroy; override;
  end;

{TNT-WARN TResourceStream}
  TTntResourceStream = class(TCustomMemoryStream)
  private
    HResInfo: HRSRC;
    HGlobal: THandle;
    procedure Initialize(Instance: THandle; Name, ResType: PWideChar);
  public
    constructor Create(Instance: THandle; const ResName: WideString; ResType: PWideChar);
    constructor CreateFromID(Instance: THandle; ResID: Word; ResType: PWideChar);
    destructor Destroy; override;
    function Write(const Buffer; Count: Longint): Longint; override;
  end;

{TNT-WARN TStrings}
  TTntWideStrings = class(TWideStrings)
  private
    FAnsiStrings: TStrings{TNT-ALLOW TStrings};
    procedure SetAnsiStrings(const Value: TStrings{TNT-ALLOW TStrings});
    procedure ReadData(Reader: TReader);
    procedure WriteData(Writer: TWriter);
    procedure ReadDataUTF8(Reader: TReader);
    procedure ReadDataUTF7(Reader: TReader);
    procedure WriteDataUTF7(Writer: TWriter);
  protected
    procedure DefineProperties(Filer: TFiler); override;
{$IFDEF JCL}
  protected
    function GetTextStr: WideString; virtual;
    procedure SetTextStr(const Value: WideString); virtual;
    procedure Put(Index: Integer; const S: WideString); override;
    procedure PutObject(Index: Integer; AObject: TObject); override;
  public
    function GetText: WideString; override;
    procedure SetText(const Value: WideString); override;
{$ENDIF}
  public
    constructor Create;
    destructor Destroy; override;
    procedure LoadFromFile(const FileName: WideString); reintroduce; virtual;
    procedure SaveToFile(const FileName: WideString); reintroduce; virtual;
  published
    property AnsiStrings: TStrings{TNT-ALLOW TStrings} read FAnsiStrings write SetAnsiStrings stored False;
  end;

{TNT-WARN TStringList}
{TNT-WARN TWideStringList}
  TTntWideStringList = class(TTntWideStrings)
  private
    FList: TWideStringList{TNT-ALLOW TWideStringList};
    function GetDuplicates: TDuplicates;
    procedure SetDuplicates(const Value: TDuplicates);
    function GetSorted: Boolean;
    procedure SetSorted(const Value: Boolean);
    function GetOnChange: TNotifyEvent;
    procedure SetOnChange(const Value: TNotifyEvent);
    function GetOnChanging: TNotifyEvent;
    procedure SetOnChanging(const Value: TNotifyEvent);
  protected
    procedure Changed; virtual;
    procedure Changing; virtual;
    function Get(Index: Integer): WideString; override;
    function GetCapacity: Integer; override;
    function GetCount: Integer; override;
    function GetObject(Index: Integer): TObject; override;
    procedure Put(Index: Integer; const S: WideString); override;
    procedure PutObject(Index: Integer; AObject: TObject); override;
    procedure SetCapacity(NewCapacity: Integer); override;
    procedure SetUpdateState(Updating: Boolean); override;
    procedure SetLanguage(Value: LCID); override;
  public
    constructor Create;
    destructor Destroy; override;
    function Add(const S: WideString): Integer; override;
    procedure Clear; override;
    procedure Delete(Index: Integer); override;
    procedure Exchange(Index1, Index2: Integer); override;
    function Find(const S: WideString; var Index: Integer): Boolean; virtual;
    function IndexOf(const S: WideString): Integer; override;
    procedure Insert(Index: Integer; const S: WideString); override;
    procedure Sort; virtual;
    property Duplicates: TDuplicates read GetDuplicates write SetDuplicates;
    property Sorted: Boolean read GetSorted write SetSorted;
    property OnChange: TNotifyEvent read GetOnChange write SetOnChange;
    property OnChanging: TNotifyEvent read GetOnChanging write SetOnChanging;
  end;

implementation

uses Consts, {$IFNDEF VER130} RTLConsts, {$ENDIF} SysConst, ComObj, Forms, Registry,
  ConvertUTF7, ShellApi;

function MakeObjectInstance(Method: TWndMethod): Pointer;
begin
{$IFDEF VER130}
  Result := Forms.MakeObjectInstance(Method);
{$ELSE}
  Result := Classes.MakeObjectInstance(Method);
{$ENDIF}
end;

procedure FreeObjectInstance(ObjectInstance: Pointer);
begin
{$IFDEF VER130}
  Forms.FreeObjectInstance(ObjectInstance);
{$ELSE}
  Classes.FreeObjectInstance(ObjectInstance);
{$ENDIF}
end;

//------------------------- Tnt Original Procs ----------------------------------

function WinCheckH(RetVal: Cardinal): Cardinal;
begin
  if RetVal = 0 then RaiseLastOSError;
  Result := RetVal;
end;

function WinCheckFileH(RetVal: Cardinal): Cardinal;
begin
  if RetVal = INVALID_HANDLE_VALUE then RaiseLastOSError;
  Result := RetVal;
end;

function WinCheckP(RetVal: Pointer): Pointer;
begin
  if RetVal = nil then RaiseLastOSError;
  Result := RetVal;
end;

{ Windows.pas doesn't declare TranslateCharsetInfo() correctly. }
function TranslateCharsetInfo(lpSrc: PDWORD; var lpCs: TCharsetInfo; dwFlags: DWORD): BOOL; stdcall; external gdi32 name 'TranslateCharsetInfo';

function FontCharSetToCodePage(FontCharSet: TFontCharSet): Cardinal;
var
  C: TCharsetInfo;
begin
  if TranslateCharsetInfo(PDWORD(FontCharSet), C, TCI_SRCCHARSET) then
    Result:=C.ciACP
  else
    raise Exception.Create('TNT Internal Error: Unexpected FontCharSet in FontCharSetToCodePage().');
end;

{$IFDEF JCL}
function CodePageToWideString(const A: AnsiString; CodePage: Cardinal; dwFlags: Cardinal = 0): WideString;
var
  Len: Integer;
begin
  // figure length
  Len := MultiByteToWideChar(CodePage, dwFlags, PAnsiChar(A), Length(A), nil, 0);
  SetLength(Result, Len);
  if Len > 0 then begin
    // convert string
    Len := MultiByteToWideChar(CodePage, dwFlags, PAnsiChar(A), Length(A),
      PWideChar(Result), Length(Result));
    // check result
    if Len = 0 then
      RaiseLastOSError
    else
      SetLength(Result, Len);
  end;
end;
{$ENDIF}

function WideStringToCodePage(const W: WideString; CodePage: Cardinal; dwFlags: Cardinal = 0): AnsiString;
var
  Len: Integer;
begin
  // figure length
  Len := WideCharToMultiByte(CodePage, dwFlags, PWideChar(W), Length(W), nil, 0, nil, nil);
  // convert string
  SetLength(Result, Len);
  if Len > 0 then begin
    Len := WideCharToMultiByte(CodePage, dwFlags, PWideChar(W), Length(W),
      PAnsiChar(Result), Length(Result), nil, nil);
    // check result
    if Len = 0 then
      RaiseLastOSError
    else
      SetLength(Result, Len);
  end;
end;

function UCS2ToWideString(const Value: AnsiString): WideString;
begin
  if Length(Value) = 0 then
    Result := ''
  else
    SetString(Result, PWideChar(@Value[1]), Length(Value) div SizeOf(WideChar))
end;

function WideStringToUCS2(const Value: WideString): AnsiString;
begin
  if Length(Value) = 0 then
    Result := ''
  else
    SetString(Result, PAnsiChar(@Value[1]), Length(Value) * SizeOf(WideChar))
end;

function WideTextPos(const SubStr, S: WideString): Integer;
begin
  Result := Pos(Unicode_WideUpperCase(SubStr), Unicode_WideUpperCase(S));
end;

function IsWideCharDigit(WC: WideChar): Boolean;
begin
  Result := UnicodeIsDigit(UCS4(WC));
end;

function IsWideCharAlpha(WC: WideChar): Boolean;
begin
  Result := UnicodeIsAlpha(UCS4(WC));
end;

function IsWideCharAlphaNumeric(WC: WideChar): Boolean;
begin
  Result := UnicodeIsAlphaNum(UCS4(WC));
end;

function WideGetModuleFileName(Instance: HModule): WideString;
var
  AnsiResult: AnsiString;
begin
  if Win32PlatformIsUnicode then begin
    SetLength(Result, MAX_PATH);
    WinCheckH(GetModuleFileNameW(Instance, PWideChar(Result), MAX_PATH));
    Result := PWideChar(Result)
  end else begin
    SetLength(AnsiResult, MAX_PATH);
    WinCheckH(GetModuleFileNameA(Instance, PAnsiChar(AnsiResult), MAX_PATH));
    Result := PAnsiChar(AnsiResult)
  end;
end;

function ClassIsRegistered(const clsid: TCLSID): Boolean;
var
  OleStr: POleStr;
  Reg: TRegIniFile;
  Key, Filename: WideString;
begin
  // First, check to see if there is a ProgID.  This will tell if the
  // control is registered on the machine.  No ProgID, control won't run
  Result := ProgIDFromCLSID(clsid, OleStr) = S_OK;
  if not Result then Exit;  //Bail as soon as anything goes wrong.

  // Next, make sure that the file is actually there by rooting it out
  // of the registry
  Key := WideFormat('\SOFTWARE\Classes\CLSID\%s', [GUIDToString(clsid)]);
  Reg := TRegIniFile.Create;
  try
    Reg.RootKey := HKEY_LOCAL_MACHINE;
    Result := Reg.OpenKeyReadOnly(Key);
    if not Result then Exit; // Bail as soon as anything goes wrong.

    FileName := Reg.ReadString('InProcServer32', '', EmptyStr);
    if (Filename = EmptyStr) then // try another key for the file name
    begin
      FileName := Reg.ReadString('InProcServer', '', EmptyStr);
    end;
    Result := Filename <> EmptyStr;
    if not Result then Exit;
    Result := WideFileExists(Filename);
  finally
    Reg.Free;
  end;
end;

//---------------- Tnt - System.pas --------------------------------------------

function WideGetParamStr(P: PWideChar; var Param: WideString): PWideChar;
var
  Len: Integer;
  Buffer: array[0..4095] of WideChar;
begin
  while True do
  begin
    while (P[0] <> #0) and (P[0] <= ' ') do Inc(P);
    if (P[0] = '"') and (P[1] = '"') then Inc(P, 2) else Break;
  end;
  Len := 0;
  while (P[0] > ' ') and (Len < SizeOf(Buffer)) do
    if P[0] = '"' then
    begin
      Inc(P);
      while (P[0] <> #0) and (P[0] <> '"') do
      begin
        Buffer[Len] := P[0];
        Inc(Len);
        Inc(P);
      end;
      if P[0] <> #0 then Inc(P);
    end else
    begin
      Buffer[Len] := P[0];
      Inc(Len);
      Inc(P);
    end;
  SetString(Param, Buffer, Len);
  Result := P;
end;

function WideParamCount: Integer;
var
  P: PWideChar;
  S: WideString;
begin
  P := WideGetParamStr(GetCommandLineW, S);
  Result := 0;
  while True do
  begin
    P := WideGetParamStr(P, S);
    if S = '' then Break;
    Inc(Result);
  end;
end;

function WideParamStr(Index: Integer): WideString;
var
  P: PWideChar;
begin
  if Index = 0 then
    Result := WideGetModuleFileName(0)
  else
  begin
    P := GetCommandLineW;
    while True do
    begin
      P := WideGetParamStr(P, Result);
      if (Index = 0) or (Result = '') then Break;
      Dec(Index);
    end;
  end;
end;

//---------------- Tnt - Windows.pas --------------------------------------------

function Tnt_CreateFileW(lpFileName: PWideChar; dwDesiredAccess, dwShareMode: DWORD;
  lpSecurityAttributes: PSecurityAttributes; dwCreationDisposition, dwFlagsAndAttributes: DWORD;
    hTemplateFile: THandle): THandle;
begin
  if Win32PlatformIsUnicode then
    Result := CreateFileW(lpFileName, dwDesiredAccess, dwShareMode,
      lpSecurityAttributes, dwCreationDisposition, dwFlagsAndAttributes, hTemplateFile)
  else
    Result := CreateFileA{TNT-ALLOW CreateFileA}(PAnsiChar(AnsiString(lpFileName)), dwDesiredAccess, dwShareMode,
      lpSecurityAttributes, dwCreationDisposition, dwFlagsAndAttributes, hTemplateFile)
end;

function Tnt_Is_IntResource(ResStr: LPCWSTR): Boolean;
begin
  Result := HiWord(Cardinal(ResStr)) = 0;
end;

procedure MakeWideWin32FindData(var WideFindData: TWIN32FindDataW; AnsiFindData: TWIN32FindDataA);
begin
  CopyMemory(@WideFindData, @AnsiFindData,
    Integer(@WideFindData.cFileName) - Integer(@WideFindData));
  StrPCopyW{TNT-ALLOW StrPCopyW}(WideFindData.cFileName, AnsiFindData.cFileName);
  StrPCopyW{TNT-ALLOW StrPCopyW}(WideFindData.cAlternateFileName, AnsiFindData.cAlternateFileName);
end;

function Tnt_FindFirstFileW(lpFileName: PWideChar; var lpFindFileData: TWIN32FindDataW): THandle;
var
  Ansi_lpFindFileData: TWIN32FindDataA;
begin
  if Win32PlatformIsUnicode then
    Result := FindFirstFileW(lpFileName, lpFindFileData)
  else begin
    Result := FindFirstFileA{TNT-ALLOW FindFirstFileA}(PAnsiChar(AnsiString(lpFileName)),
      Ansi_lpFindFileData);
    if Result <> INVALID_HANDLE_VALUE then
      MakeWideWin32FindData(lpFindFileData, Ansi_lpFindFileData);
  end;
end;

function Tnt_FindNextFileW(hFindFile: THandle; var lpFindFileData: TWIN32FindDataW): BOOL;
var
  Ansi_lpFindFileData: TWIN32FindDataA;
begin
  if Win32PlatformIsUnicode then
    Result := FindNextFileW(hFindFile, lpFindFileData)
  else begin
    Result := FindNextFileA{TNT-ALLOW FindNextFileA}(hFindFile, Ansi_lpFindFileData);
    if Result then
      MakeWideWin32FindData(lpFindFileData, Ansi_lpFindFileData);
  end;
end;

function Tnt_DrawTextW(hDC: HDC; lpString: PWideChar; nCount: Integer;
  var lpRect: TRect; uFormat: UINT): Integer;
begin
  if Win32PlatformIsUnicode then
    Result := DrawTextW(hDC, lpString, nCount, lpRect, uFormat)
  else
    Result := DrawTextA{TNT-ALLOW DrawTextA}(hDC, PAnsiChar(AnsiString(lpString)),
      Length(AnsiString(Copy(WideString(lpString), 1, nCount))),
        lpRect, uFormat);
end;

function Tnt_GetDiskFreeSpaceW(lpRootPathName: PWideChar; var lpSectorsPerCluster,
  lpBytesPerSector, lpNumberOfFreeClusters, lpTotalNumberOfClusters: DWORD): BOOL;
begin
  if Win32PlatformIsUnicode then
    Result := GetDiskFreeSpaceW(lpRootPathName,
      lpSectorsPerCluster, lpBytesPerSector, lpNumberOfFreeClusters, lpTotalNumberOfClusters)
  else
    Result := GetDiskFreeSpaceA{TNT-ALLOW GetDiskFreeSpaceA}(PAnsiChar(AnsiString(lpRootPathName)),
      lpSectorsPerCluster, lpBytesPerSector, lpNumberOfFreeClusters, lpTotalNumberOfClusters)
end;

function MyPAnsiCharWithNil(const S: AnsiString): PAnsiChar;
begin
  if S = '' then
    Result := nil {Win9x needs nil for some parameters instead of empty strings}
  else
    Result := PAnsiChar(S);
end;

function MyPWideCharWithNil(const S: WideString): PWideChar;
begin
  if S = '' then
    Result := nil {Win9x needs nil for some parameters instead of empty strings}
  else
    Result := PWideChar(S);
end;

function Tnt_ShellExecuteW(hWnd: HWND; Operation, FileName, Parameters,
  Directory: PWideChar; ShowCmd: Integer): HINST;
begin
  if Win32PlatformIsUnicode then
    Result := ShellExecuteW(hWnd, MyPWideCharWithNil(WideString(Operation)),
      FileName, Parameters,
        Directory, ShowCmd)
  else begin
    Result := ShellExecuteA{TNT-ALLOW ShellExecuteA}(hWnd, MyPAnsiCharWithNil(AnsiString(Operation)),
      MyPAnsiCharWithNil(AnsiString(FileName)), MyPAnsiCharWithNil(AnsiString(Parameters)),
        MyPAnsiCharWithNil(AnsiString(Directory)), ShowCmd)
  end;
end;

function Tnt_LoadLibraryExW(lpLibFileName: PWideChar; hFile: THandle; dwFlags: DWORD): HMODULE;
begin
  if Win32PlatformIsUnicode then
    Result := LoadLibraryExW(lpLibFileName, hFile, dwFlags)
  else
    Result := LoadLibraryExA{TNT-ALLOW LoadLibraryExA}(PAnsiChar(AnsiString(lpLibFileName)), hFile, dwFlags);
end;

function Tnt_CreateProcessW(lpApplicationName: PWideChar; lpCommandLine: PWideChar;
  lpProcessAttributes, lpThreadAttributes: PSecurityAttributes;
    bInheritHandles: BOOL; dwCreationFlags: DWORD; lpEnvironment: Pointer;
      lpCurrentDirectory: PWideChar; const lpStartupInfo: TStartupInfo;
        var lpProcessInformation: TProcessInformation): BOOL;
begin
  if Win32PlatformIsUnicode then begin
    Result := CreateProcessW(lpApplicationName, lpCommandLine,
      lpProcessAttributes, lpThreadAttributes, bInheritHandles, dwCreationFlags, lpEnvironment,
        lpCurrentDirectory, lpStartupInfo, lpProcessInformation)
  end else begin
    Result := CreateProcessA{TNT-ALLOW CreateProcessA}(MyPAnsiCharWithNil(AnsiString(lpApplicationName)),
      MyPAnsiCharWithNil(AnsiString(lpCommandLine)),
        lpProcessAttributes, lpThreadAttributes, bInheritHandles, dwCreationFlags, lpEnvironment,
          MyPAnsiCharWithNil(AnsiString(lpCurrentDirectory)), lpStartupInfo, lpProcessInformation);
  end;
end;

{$IFDEF VER130}
procedure RaiseLastOSError;
begin
  RaiseLastWin32Error;
end;

function WideFormat(const FormatStr: WideString; const Args: array of const): WideString;
begin
  Result := Format{TNT-ALLOW Format}(FormatStr, Args);
end;

function WideCompareStr(const W1, W2: WideString): Integer;
begin
  if Win32PlatformIsUnicode then
    Result := CompareStringW(LOCALE_USER_DEFAULT, 0,
      PWideChar(W1), Length(W1), PWideChar(W2), Length(W2)) - 2
  else
    Result := AnsiCompareStr{TNT-ALLOW AnsiCompareStr}(W1, W2);
end;

function WideSameStr(const W1, W2: WideString): Boolean;
begin
  Result := WideCompareStr(W1, W2) = 0;
end;

function WideSameText(const W1, W2: WideString): Boolean;
begin
  Result := WideCompareText(W1, W2) = 0;
end;
{$ENDIF}

function WideCompareText(const W1, W2: WideString): Integer;
begin
{$IFDEF VER130}
  if Win32PlatformIsUnicode then
    Result := CompareStringW(LOCALE_USER_DEFAULT, NORM_IGNORECASE,
      PWideChar(W1), Length(W1), PWideChar(W2), Length(W2)) - 2
  else
    Result := AnsiCompareText{TNT-ALLOW AnsiCompareText}(W1, W2);
{$ELSE}
  Result := SysUtils.WideCompareText(W1, W2);
{$ENDIF}
end;

function Unicode_WideLowerCase(const S: WideString): WideString;
begin
{The version in SysUtils is broken for Win 9X.}
{$IFDEF JCL}
  Result := JclUnicode.WideLowerCase{TNT-ALLOW WideLowerCase}(S);
{$ELSE}
  Result := Unicode.WideLowerCase{TNT-ALLOW WideLowerCase}(S);
{$ENDIF}
end;

function Unicode_WideUpperCase(const S: WideString): WideString;
begin
{The version in SysUtils is broken for Win 9X.}
{$IFDEF JCL}
  Result := JclUnicode.WideUpperCase{TNT-ALLOW WideUpperCase}(S);
{$ELSE}
  Result := Unicode.WideUpperCase{TNT-ALLOW WideUpperCase}(S);
{$ENDIF}
end;

function WideLastChar(W: WideString): WideChar;
begin
  if Length(W) = 0 then
    Result := #0
  else
    Result := W[Length(W)];
end;

function WideStringReplace(const S, OldPattern, NewPattern: WideString;
  Flags: TReplaceFlags; WholeWord: Boolean = False): WideString;

  function IsWordSeparator(WC: WideChar): Boolean;
  begin
    Result := (WC = WideChar(#0))
           or UnicodeIsWhiteSpace(Cardinal(WC))
           or UnicodeIsPunctuation(Cardinal(WC));
  end;

var
  SearchStr, Patt, NewStr: WideString;
  Offset: Integer;
  PrevChar, NextChar: WideChar;
begin
  if rfIgnoreCase in Flags then
  begin
    SearchStr := Unicode_WideUpperCase(S);
    Patt := Unicode_WideUpperCase(OldPattern);
  end else
  begin
    SearchStr := S;
    Patt := OldPattern;
  end;
  NewStr := S;
  Result := '';
  while SearchStr <> '' do
  begin
    Offset := Pos(Patt, SearchStr);
    if Offset = 0 then
    begin
      Result := Result + NewStr;
      Break;
    end; // done

    if (WholeWord) then
    begin
      if (Offset = 1) then
        PrevChar := WideLastChar(Result)
      else
        PrevChar := NewStr[Offset - 1];

      if Offset + Length(OldPattern) <= Length(NewStr) then
        NextChar := NewStr[Offset + Length(OldPattern)]
      else
        NextChar := WideChar(#0);

      if (not IsWordSeparator(PrevChar))
      or (not IsWordSeparator(NextChar)) then
      begin
        Result := Result + Copy(NewStr, 1, Offset + Length(OldPattern) - 1);
        NewStr := Copy(NewStr, Offset + Length(OldPattern), MaxInt);
        SearchStr := Copy(SearchStr, Offset + Length(Patt), MaxInt);
        continue;
      end;
    end;

    Result := Result + Copy(NewStr, 1, Offset - 1) + NewPattern;
    NewStr := Copy(NewStr, Offset + Length(OldPattern), MaxInt);
    if not (rfReplaceAll in Flags) then
    begin
      Result := Result + NewStr;
      Break;
    end;
    SearchStr := Copy(SearchStr, Offset + Length(Patt), MaxInt);
  end;
end;

function TntAdjustLineBreaksLength(const S: WideString; Style: TTntTextLineBreakStyle = tlbsCRLF): Integer;
var
  Source, SourceEnd: PWideChar;
begin
  Source := Pointer(S);
  SourceEnd := Source + Length(S);
  Result := Length(S);
  while Source < SourceEnd do
  begin
    case Source^ of
      #10, LineSeparator:
        if Style = tlbsCRLF then
          Inc(Result);
      #13:
        if Style = tlbsCRLF then
          if Source[1] = #10 then
            Inc(Source)
          else
            Inc(Result)
        else
          if Source[1] = #10 then
            Dec(Result);
    end;
    Inc(Source);
  end;
end;

function TntAdjustLineBreaks(const S: WideString; Style: TTntTextLineBreakStyle = tlbsCRLF): WideString;
var
  Source, SourceEnd, Dest: PWideChar;
  DestLen: Integer;
begin
  Source := Pointer(S);
  SourceEnd := Source + Length(S);
  DestLen := TntAdjustLineBreaksLength(S, Style);
  SetString(Result, nil, DestLen);
  Dest := Pointer(Result);
  while Source < SourceEnd do begin
    case Source^ of
      #10, LineSeparator:
        begin
          if Style in [tlbsCRLF, tlbsCR] then
          begin
            Dest^ := #13;
            Inc(Dest);
          end;
          if Style in [tlbsCRLF, tlbsLF] then
          begin
            Dest^ := #10;
            Inc(Dest);
          end;
          Inc(Source);
        end;
      #13:
        begin
          if Style in [tlbsCRLF, tlbsCR] then
          begin
            Dest^ := #13;
            Inc(Dest);
          end;
          if Style in [tlbsCRLF, tlbsLF] then
          begin
            Dest^ := #10;
            Inc(Dest);
          end;
          Inc(Source);
          if Source^ = #10 then Inc(Source);
        end;
    else
      Dest^ := Source^;
      Inc(Dest);
      Inc(Source);
    end;
  end;
end;

function WideWrapText(const Line, BreakStr: WideString; const BreakChars: TSysCharSet;
  MaxCol: Integer): WideString;

  function WideCharIn(C: WideChar; SysCharSet: TSysCharSet): Boolean;
  begin
    Result := (C <= High(AnsiChar)) and (AnsiChar(C) in SysCharSet);
  end;

const
  QuoteChars = ['''', '"'];
var
  Col, Pos: Integer;
  LinePos, LineLen: Integer;
  BreakLen, BreakPos: Integer;
  QuoteChar, CurChar: WideChar;
  ExistingBreak: Boolean;
begin
  Col := 1;
  Pos := 1;
  LinePos := 1;
  BreakPos := 0;
  QuoteChar := ' ';
  ExistingBreak := False;
  LineLen := Length(Line);
  BreakLen := Length(BreakStr);
  Result := '';
  while Pos <= LineLen do
  begin
    CurChar := Line[Pos];
    if CurChar = BreakStr[1] then
    begin
      if QuoteChar = ' ' then
      begin
        ExistingBreak := WideCompareText(BreakStr, Copy(Line, Pos, BreakLen)) = 0;
        if ExistingBreak then
        begin
          Inc(Pos, BreakLen-1);
          BreakPos := Pos;
        end;
      end
    end
    else if WideCharIn(CurChar, BreakChars) then
    begin
      if QuoteChar = ' ' then BreakPos := Pos
    end
    else if WideCharIn(CurChar, QuoteChars) then
    begin
      if CurChar = QuoteChar then
        QuoteChar := ' '
      else if QuoteChar = ' ' then
        QuoteChar := CurChar;
    end;
    Inc(Pos);
    Inc(Col);
    if not (WideCharIn(QuoteChar, QuoteChars)) and (ExistingBreak or
      ((Col > MaxCol) and (BreakPos > LinePos))) then
    begin
      Col := Pos - BreakPos;
      Result := Result + Copy(Line, LinePos, BreakPos - LinePos + 1);
      if not (WideCharIn(CurChar, QuoteChars)) then
        while Pos <= LineLen do
        begin
          if WideCharIn(Line[Pos], BreakChars) then
            Inc(Pos)
          else if Copy(Line, Pos, Length(sLineBreak)) = sLineBreak then
            Inc(Pos, Length(sLineBreak))
          else
            break;
        end;
      if not ExistingBreak and (Pos < LineLen) then
        Result := Result + BreakStr;
      Inc(BreakPos);
      LinePos := BreakPos;
      ExistingBreak := False;
    end;
  end;
  Result := Result + Copy(Line, LinePos, MaxInt);
end;

function WideWrapText(const Line: WideString; MaxCol: Integer): WideString;
begin
  Result := WideWrapText(Line, sLineBreak, [' ', '-', #9], MaxCol); { do not localize }
end;

function WideFileCreate(const FileName: WideString): Integer;
begin
  Result := Integer(Tnt_CreateFileW(PWideChar(FileName), GENERIC_READ or GENERIC_WRITE,
    0, nil, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, 0))
end;

function WideFileOpen(const FileName: WideString; Mode: LongWord): Integer;
const
  AccessMode: array[0..2] of LongWord = (
    GENERIC_READ,
    GENERIC_WRITE,
    GENERIC_READ or GENERIC_WRITE);
  ShareMode: array[0..4] of LongWord = (
    0,
    0,
    FILE_SHARE_READ,
    FILE_SHARE_WRITE,
    FILE_SHARE_READ or FILE_SHARE_WRITE);
begin
  Result := Integer(Tnt_CreateFileW(PWideChar(FileName), AccessMode[Mode and 3],
    ShareMode[(Mode and $F0) shr 4], nil, OPEN_EXISTING,
      FILE_ATTRIBUTE_NORMAL, 0));
end;

function IntValidDateTimeStr(Str: WideString; Flags: Integer): Boolean;
var
  TheDateTime: Double;
begin
  Result := Succeeded(VarDateFromStr(Str, GetThreadLocale, Flags, TheDateTime));
end;

function ValidDateTimeStr(Str: WideString): Boolean;
begin
  Result := IntValidDateTimeStr(Str, 0);
end;

function ValidDateStr(Str: WideString): Boolean;
begin
  Result := IntValidDateTimeStr(Str, VAR_DATEVALUEONLY);
end;

function ValidTimeStr(Str: WideString): Boolean;
begin
  Result := IntValidDateTimeStr(Str, VAR_TIMEVALUEONLY);
end;

function IntStrToDateTime(Str: WideString; Flags: Integer; ErrorFormatStr: WideString): TDateTime;
var
  TheDateTime: Double;
begin
  try
    OleCheck(VarDateFromStr(Str, GetThreadLocale, Flags, TheDateTime));
    Result := TheDateTime;
  except
    on E: Exception do begin
      E.Message := E.Message + CRLF + WideFormat(ErrorFormatStr, [Str]);
      raise EConvertError.Create(E.Message);
    end;
  end;
end;

function TntStrToDateTime(Str: WideString): TDateTime;
begin
  Result := IntStrToDateTime(Str, 0, SInvalidDateTime);
end;

function TntStrToDate(Str: WideString): TDateTime;
begin
  Result := IntStrToDateTime(Str, VAR_DATEVALUEONLY, SInvalidDate);
end;

function TntStrToTime(Str: WideString): TTime;
begin
  Result := IntStrToDateTime(Str, VAR_TIMEVALUEONLY, SInvalidTime);
end;

function WideFindMatchingFile(var F: TSearchRecW): Integer;
var
  LocalFileTime: TFileTime;
begin
  with F do
  begin
    while FindData.dwFileAttributes and ExcludeAttr <> 0 do
      if not Tnt_FindNextFileW(FindHandle, FindData) then
      begin
        Result := GetLastError;
        Exit;
      end;
    FileTimeToLocalFileTime(FindData.ftLastWriteTime, LocalFileTime);
    FileTimeToDosDateTime(LocalFileTime, LongRec(Time).Hi, LongRec(Time).Lo);
    Size := FindData.nFileSizeLow;
    Attr := FindData.dwFileAttributes;
    Name := FindData.cFileName;
  end;
  Result := 0;
end;

function WideFindFirst(const Path: WideString; Attr: Integer; var F: TSearchRecW): Integer;
const
  faSpecial = faHidden or faSysFile or faVolumeID or faDirectory;
begin
  F.ExcludeAttr := not Attr and faSpecial;
  F.FindHandle := Tnt_FindFirstFileW(PWideChar(Path), F.FindData);
  if F.FindHandle <> INVALID_HANDLE_VALUE then
  begin
    Result := WideFindMatchingFile(F);
    if Result <> 0 then WideFindClose(F);
  end else
    Result := GetLastError;
end;

function WideFindNext(var F: TSearchRecW): Integer;
begin
  if Tnt_FindNextFileW(F.FindHandle, F.FindData) then
    Result := WideFindMatchingFile(F) else
    Result := GetLastError;
end;

procedure WideFindClose(var F: TSearchRecW);
begin
  if F.FindHandle <> INVALID_HANDLE_VALUE then
  begin
    Windows.FindClose(F.FindHandle);
    F.FindHandle := INVALID_HANDLE_VALUE;
  end;
end;

function WideDirectoryExists(const Name: WideString): Boolean;
var
  Code: Cardinal;
begin
  Code := WideFileGetAttr(Name);
  Result := (Integer(Code) <> -1) and (FILE_ATTRIBUTE_DIRECTORY and Code <> 0);
end;

function WideFileExists(const Name: WideString): Boolean;
var
  Code: Cardinal;
begin
  Code := WideFileGetAttr(Name);
  Result := (Integer(Code) <> -1) and ((FILE_ATTRIBUTE_DIRECTORY and Code) = 0);
end;

function WideFileGetAttr(const FileName: WideString): Cardinal;
begin
  if Win32PlatformIsUnicode then
    Result := GetFileAttributesW(PWideChar(FileName))
  else
    Result := GetFileAttributesA(PAnsiChar(AnsiString(FileName)))
end;

function WideFileSetAttr(const FileName: WideString; Attr: Integer): Boolean;
begin
  if Win32PlatformIsUnicode then
    Result := SetFileAttributesW(PWideChar(FileName), Attr)
  else
    Result := SetFileAttributesA(PAnsiChar(AnsiString(FileName)), Attr)
end;

function WideIsPathDelimiter(const S: WideString; Index: Integer): Boolean;
begin
  Result := (Index > 0) and (Index <= Length(S)) and (S[Index] = PathDelim);
end;

function WideIsDelimiter(const Delimiters, S: WideString; Index: Integer): Boolean;
begin
  Result := False;
  if (Index <= 0) or (Index > Length(S)) then exit;
  Result := StrScanW(PWideChar(Delimiters), S[Index]) <> nil;
end;

function WideIncludeTrailingBackslash(const S: WideString): WideString;
begin
  Result := S;
  if not WideIsPathDelimiter(Result, Length(Result)) then Result := Result + PathDelim;
end;

function WideExcludeTrailingBackslash(const S: WideString): WideString;
begin
  Result := S;
  if WideIsPathDelimiter(Result, Length(Result)) then
    SetLength(Result, Length(Result)-1);
end;

function WideLastDelimiter(const Delimiters, S: WideString): Integer;
var
  P: PWideChar;
begin
  Result := Length(S);
  P := PWideChar(Delimiters);
  while Result > 0 do
  begin
    if (S[Result] <> #0) and (StrScanW(P, S[Result]) <> nil) then
      Exit;
    Dec(Result);
  end;
end;

function WideChangeFileExt(const FileName, Extension: WideString): WideString;
var
  I: Integer;
begin
  I := WideLastDelimiter('.\:',Filename);
  if (I = 0) or (FileName[I] <> '.') then I := MaxInt;
  Result := Copy(FileName, 1, I - 1) + Extension;
end;

function WideExtractFilePath(const FileName: WideString): WideString;
var
  I: Integer;
begin
  I := WideLastDelimiter('\:', FileName);
  Result := Copy(FileName, 1, I);
end;

function WideExtractFileDir(const FileName: WideString): WideString;
var
  I: Integer;
begin
  I := WideLastDelimiter(DriveDelim + PathDelim,Filename);
  if (I > 1) and (FileName[I] = PathDelim) and
    (not (FileName[I - 1] in [WideChar(PathDelim), WideChar(DriveDelim)])) then Dec(I);
  Result := Copy(FileName, 1, I);
end;

function WideExtractFileDrive(const FileName: WideString): WideString;
var
  I, J: Integer;
begin
  if (Length(FileName) >= 2) and (FileName[2] = DriveDelim) then
    Result := Copy(FileName, 1, 2)
  else if (Length(FileName) >= 2) and (FileName[1] = PathDelim) and
    (FileName[2] = PathDelim) then
  begin
    J := 0;
    I := 3;
    While (I < Length(FileName)) and (J < 2) do
    begin
      if FileName[I] = PathDelim then Inc(J);
      if J < 2 then Inc(I);
    end;
    if FileName[I] = PathDelim then Dec(I);
    Result := Copy(FileName, 1, I);
  end else Result := '';
end;

function WideExtractFileName(const FileName: WideString): WideString;
var
  I: Integer;
begin
  I := WideLastDelimiter('\:', FileName);
  Result := Copy(FileName, I + 1, MaxInt);
end;

function WideExtractFileExt(const FileName: WideString): WideString;
var
  I: Integer;
begin
  I := WideLastDelimiter('.\:', FileName);
  if (I > 0) and (FileName[I] = '.') then
    Result := Copy(FileName, I, MaxInt) else
    Result := '';
end;

function WideForceDirectories(Dir: WideString): Boolean;
begin
  Result := True;
  if Length(Dir) = 0 then
    raise Exception.Create(SCannotCreateDir);
  Dir := WideExcludeTrailingBackslash(Dir);
  if (Length(Dir) < 3) or WideDirectoryExists(Dir)
    or (WideExtractFilePath(Dir) = Dir) then Exit; // avoid 'xyz:\' problem.
  Result := WideForceDirectories(WideExtractFilePath(Dir));
  if Result then begin
    if Win32PlatformIsUnicode then
      Result := CreateDirectoryW(PWideChar(Dir), nil)
    else
      Result := CreateDirectoryA(PAnsiChar(AnsiString(Dir)), nil)
  end;
end;

function WideFileSearch(const Name, DirList: WideString): WideString;
var
  I, P, L: Integer;
  C: WideChar;
begin
  Result := Name;
  P := 1;
  L := Length(DirList);
  while True do
  begin
    if WideFileExists(Result) then Exit;
    while (P <= L) and (DirList[P] = PathSep) do Inc(P);
    if P > L then Break;
    I := P;
    while (P <= L) and (DirList[P] <> PathSep) do
      Inc(P);
    Result := Copy(DirList, I, P - I);
    C := WideLastChar(Result);
    if (C <> DriveDelim) and (C <> PathDelim) then
      Result := Result + PathDelim;
    Result := Result + Name;
  end;
  Result := '';
end;


function WideExtractRelativePath(const BaseName, DestName: WideString): WideString;
var
  BasePath, DestPath: WideString;
  BaseLead, DestLead: PWideChar;
  BasePtr, DestPtr: PWideChar;

  function WideExtractFilePathNoDrive(const FileName: WideString): WideString;
  begin
    Result := WideExtractFilePath(FileName);
    Delete(Result, 1, Length(WideExtractFileDrive(FileName)));
  end;

  function Next(var Lead: PWideChar): PWideChar;
  begin
    Result := Lead;
    if Result = nil then Exit;
    Lead := StrScanW(Lead, PathDelim);
    if Lead <> nil then
    begin
      Lead^ := #0;
      Inc(Lead);
    end;
  end;

begin
  if WideSameText(WideExtractFileDrive(BaseName), WideExtractFileDrive(DestName)) then
  begin
    BasePath := WideExtractFilePathNoDrive(BaseName);
    DestPath := WideExtractFilePathNoDrive(DestName);
    BaseLead := Pointer(BasePath);
    BasePtr := Next(BaseLead);
    DestLead := Pointer(DestPath);
    DestPtr := Next(DestLead);
    while (BasePtr <> nil) and (DestPtr <> nil) and WideSameText(BasePtr, DestPtr) do
    begin
      BasePtr := Next(BaseLead);
      DestPtr := Next(DestLead);
    end;
    Result := '';
    while BaseLead <> nil do
    begin
      Result := Result + '..' + PathDelim;             { Do not localize }
      Next(BaseLead);
    end;
    if (DestPtr <> nil) and (DestPtr^ <> #0) then
      Result := Result + DestPtr + PathDelim;
    if DestLead <> nil then
      Result := Result + DestLead;     // destlead already has a trailing backslash
    Result := Result + WideExtractFileName(DestName);
  end
  else
    Result := DestName;
end;

function WideRenameFile(const OldName, NewName: WideString): Boolean;
begin
  if Win32PlatformIsUnicode then
    Result := MoveFileW(PWideChar(OldName), PWideChar(NewName))
  else
    Result := MoveFileA(PAnsiChar(AnsiString(OldName)), PAnsiChar(AnsiString(NewName)))
end;

function WideDeleteFile(const FileName: WideString): Boolean;
begin
  if Win32PlatformIsUnicode	then
    Result := DeleteFileW(PWideChar(FileName))
  else
    Result := DeleteFileA(PAnsiChar(AnsiString(FileName)))
end;

function WideCopyFile(FromFile, ToFile: WideString; FailIfExists: Boolean): Boolean;
begin
  if Win32PlatformIsUnicode then
    Result := CopyFileW(PWideChar(FromFile), PWideChar(ToFile), FailIfExists)
  else
    Result := CopyFileA(PAnsiChar(AnsiString(FromFile)), PAnsiChar(AnsiString(ToFile)), FailIfExists)
end;

{ TTntFileStream }

constructor TTntFileStream.Create(const FileName: WideString; Mode: Word);
var
  CreateHandle: Integer;
begin
  if Mode = fmCreate then
  begin
    CreateHandle := WideFileCreate(FileName);
    if CreateHandle < 0 then
      raise EFCreateError.CreateResFmt(PResStringRec(@SFCreateError), [FileName]);
  end else
  begin
    CreateHandle := WideFileOpen(FileName, Mode);
    if CreateHandle < 0 then
      raise EFOpenError.CreateResFmt(PResStringRec(@SFOpenError), [FileName]);
  end;
  inherited Create(CreateHandle);
end;

destructor TTntFileStream.Destroy;
begin
  if Handle >= 0 then FileClose(Handle);
end;

{ TTntResourceStream }

constructor TTntResourceStream.Create(Instance: THandle; const ResName: WideString;
  ResType: PWideChar);
begin
  inherited Create;
  Initialize(Instance, PWideChar(ResName), ResType);
end;

constructor TTntResourceStream.CreateFromID(Instance: THandle; ResID: Word;
  ResType: PWideChar);
begin
  inherited Create;
  Initialize(Instance, PWideChar(ResID), ResType);
end;

procedure TTntResourceStream.Initialize(Instance: THandle; Name, ResType: PWideChar);

  procedure Error;
  begin
    raise EResNotFound.CreateFmt(SResNotFound, [Name]);
  end;

begin
  HResInfo := FindResourceW(Instance, Name, ResType);
  if HResInfo = 0 then Error;
  HGlobal := LoadResource(Instance, HResInfo);
  if HGlobal = 0 then Error;
  SetPointer(LockResource(HGlobal), SizeOfResource(Instance, HResInfo));
end;

destructor TTntResourceStream.Destroy;
begin
  UnlockResource(HGlobal);
  FreeResource(HGlobal);
  inherited Destroy;
end;

function TTntResourceStream.Write(const Buffer; Count: Longint): Longint;
begin
  raise EStreamError.CreateRes(PResStringRec(@SCantWriteResourceStreamError));
end;

{ TAnsiStringsForWideStrings }

type
  TAnsiStringsForWideStrings = class(TStrings{TNT-ALLOW TStrings})
  private
    FWideStrings: TTntWideStrings;
  protected
    function Get(Index: Integer): AnsiString; override;
    procedure Put(Index: Integer; const S: AnsiString); override;
    function GetCount: Integer; override;
    function GetObject(Index: Integer): TObject; override;
    procedure PutObject(Index: Integer; AObject: TObject); override;
    procedure SetUpdateState(Updating: Boolean); override;
  public
    procedure Clear; override;
    procedure Delete(Index: Integer); override;
    procedure Insert(Index: Integer; const S: AnsiString); override;
  end;

procedure TAnsiStringsForWideStrings.Clear;
begin
  FWideStrings.Clear;
end;

procedure TAnsiStringsForWideStrings.Delete(Index: Integer);
begin
  FWideStrings.Delete(Index);
end;

function TAnsiStringsForWideStrings.Get(Index: Integer): AnsiString;
begin
  Result := FWideStrings.Get(Index);
end;

procedure TAnsiStringsForWideStrings.Put(Index: Integer; const S: AnsiString);
begin
  FWideStrings.Put(Index, S);
end;

function TAnsiStringsForWideStrings.GetCount: Integer;
begin
  Result := FWideStrings.GetCount;
end;

procedure TAnsiStringsForWideStrings.Insert(Index: Integer; const S: AnsiString);
begin
  FWideStrings.Insert(Index, S);
end;

function TAnsiStringsForWideStrings.GetObject(Index: Integer): TObject;
begin
  Result := FWideStrings.GetObject(Index);
end;

procedure TAnsiStringsForWideStrings.PutObject(Index: Integer; AObject: TObject);
begin
  FWideStrings.PutObject(Index, AObject);
end;

procedure TAnsiStringsForWideStrings.SetUpdateState(Updating: Boolean);
begin
  FWideStrings.SetUpdateState(Updating);
end;

{ TTntWideStrings }

constructor TTntWideStrings.Create;
begin
  inherited;
  FAnsiStrings := TAnsiStringsForWideStrings.Create;
  TAnsiStringsForWideStrings(FAnsiStrings).FWideStrings := Self;
end;

destructor TTntWideStrings.Destroy;
begin
  FreeAndNil(FAnsiStrings);
  inherited;
end;

procedure TTntWideStrings.SetAnsiStrings(const Value: TStrings{TNT-ALLOW TStrings});
begin
  Clear;
  AddStrings(Value);
end;

{$IFDEF JCL}
function TTntWideStrings.GetText: WideString;
begin
  Result := GetTextStr;
end;

procedure TTntWideStrings.SetText(const Value: WideString);
begin
  SetTextStr(Value);
end;

function TTntWideStrings.GetTextStr: WideString;
begin
  Result := GetSeparatedText(WideCRLF);
end;

procedure TTntWideStrings.SetTextStr(const Value: WideString);
begin
  inherited SetText(Value);
end;

procedure TTntWideStrings.PutObject(Index: Integer; AObject: TObject);
begin
  // do nothing.
end;

procedure TTntWideStrings.Put(Index: Integer; const S: WideString);
var
  TempObject: TObject;
begin
  TempObject := GetObject(Index);
  Delete(Index);
  InsertObject(Index, S, TempObject);
end;

{$ENDIF}

procedure TTntWideStrings.ReadData(Reader: TReader);
begin
  if Reader.NextValue in [vaString, vaLString] then
    SetTextStr(Reader.ReadString) {JCL compatiblity}
  else if Reader.NextValue = vaWString then
    SetTextStr(Reader.ReadWideString) {JCL compatiblity}
  else begin
    BeginUpdate;
    try
      Clear;
      Reader.ReadListBegin;
      while not Reader.EndOfList do
        if Reader.NextValue in [vaString, vaLString] then
          Add(Reader.ReadString) {TStrings compatiblity}
        else
          Add(Reader.ReadWideString);
      Reader.ReadListEnd;
    finally
      EndUpdate;
    end;
  end;
end;

procedure TTntWideStrings.WriteData(Writer: TWriter);
var
  I: Integer;
begin
  Writer.WriteListBegin;
  for I := 0 to Count-1 do
    Writer.WriteWideString(Get(I));
  Writer.WriteListEnd;
end;

{$IFDEF VER130}
const IsDelphi5 = True;
{$ELSE}
const IsDelphi5 = False;
{$ENDIF}

procedure TTntWideStrings.ReadDataUTF8(Reader: TReader);
begin
  Reader.ReadListBegin;
  if IsDelphi5  { Delphi 5 always needs UTF help }
  or (csDesigning in Reader.Owner.ComponentState) { Designtime always needs UTF help }
  or (Count = 0){ Legacy support where 'WideStrings' was never written in lieu of WideStringsW }
  then begin
    BeginUpdate;
    try
      Clear;
      while not Reader.EndOfList do
        Add(UTF8ToWideString(Reader.ReadString))
    finally
      EndUpdate;
    end;
  end else begin
    while not Reader.EndOfList do
      Reader.ReadString; { do nothing with Result }
  end;
  Reader.ReadListEnd;
end;

procedure TTntWideStrings.ReadDataUTF7(Reader: TReader);
begin
  Reader.ReadListBegin;
  if IsDelphi5  { Delphi 5 always needs UTF help }
  or (csDesigning in Reader.Owner.ComponentState) { Designtime always needs UTF help }
  then begin
    BeginUpdate;
    try
      Clear;
      while not Reader.EndOfList do
        Add(UTF7ToWideString(Reader.ReadString))
    finally
      EndUpdate;
    end;
  end else begin
    while not Reader.EndOfList do
      Reader.ReadString; { do nothing with Result }
  end;
  Reader.ReadListEnd;
end;

procedure TTntWideStrings.WriteDataUTF7(Writer: TWriter);
var
  I: Integer;
begin
  Writer.WriteListBegin;
  for I := 0 to Count-1 do
    Writer.WriteString(WideStringToUTF7(Get(I)));
  Writer.WriteListEnd;
end;

procedure TTntWideStrings.DefineProperties(Filer: TFiler);

  function DoWrite: Boolean;
  begin
    if Filer.Ancestor <> nil then
    begin
      Result := True;
      if Filer.Ancestor is TWideStrings then Result := not Equals(TWideStrings(Filer.Ancestor))
    end
    else Result := Count > 0;
  end;

  function DoWriteAsUTF7: Boolean;
  var
    i: integer;
  begin
    Result := False;
    for i := 0 to Count - 1 do begin
      if (Strings[i] <> '') and (WideStringToUTF8(Strings[i]) <> Strings[i]) then begin
        Result := True;
        break; { found a string with non-ASCII chars (> 127) }
      end;
    end;
  end;

var
  _DoWrite: Boolean;
begin
  _DoWrite := DoWrite;
  Filer.DefineProperty('Strings', ReadData, nil, False); { to be compatible with TStrings }
  Filer.DefineProperty('WideStrings', ReadData, WriteData, _DoWrite);
  Filer.DefineProperty('WideStringsW', ReadDataUTF8, nil, False);
  Filer.DefineProperty('WideStrings_UTF7', ReadDataUTF7, WriteDataUTF7, _DoWrite and DoWriteAsUTF7);
end;

procedure TTntWideStrings.LoadFromFile(const FileName: WideString);
var
  Stream: TStream;
begin
  try
    Stream := TTntFileStream.Create(FileName, fmOpenRead or fmShareDenyNone);
    try
      LoadFromStream(Stream);
    finally
      Stream.Free;
    end;
  except
    RaiseLastOSError;
  end;
end;

procedure TTntWideStrings.SaveToFile(const FileName: WideString);
var
  Stream: TStream;
begin
  Stream := TTntFileStream.Create(FileName, fmCreate);
  try
    SaveToStream(Stream);
  finally
    Stream.Free;
  end;
end;

{ TTntWideStringListHelper }
type
  TTntWideStringListHelper = class(TWideStringList{TNT-ALLOW TWideStringList})
  private
    Parent: TTntWideStringList;
    procedure InheritedChanged;
    procedure InheritedChanging;
    function InheritedFind(const S: WideString; var Index: Integer): Boolean; 
    procedure InheritedSort;
  protected
    procedure Changed; override;
    procedure Changing; override;
  public
    function Find(const S: WideString; var Index: Integer): Boolean; override;
    procedure Sort; override;
  end;

procedure TTntWideStringListHelper.InheritedChanged;
begin
  inherited Changed;
end;

procedure TTntWideStringListHelper.Changed;
begin
  if Parent <> nil then
    Parent.Changed;
end;

procedure TTntWideStringListHelper.InheritedChanging;
begin
  inherited Changing;
end;

procedure TTntWideStringListHelper.Changing;
begin
  if Parent <> nil then
    Parent.Changing;
end;

function TTntWideStringListHelper.InheritedFind(const S: WideString; var Index: Integer): Boolean;
begin
  Result := inherited Find(S, Index);
end;

function TTntWideStringListHelper.Find(const S: WideString; var Index: Integer): Boolean;
begin
  if Parent <> nil then
    Result := Parent.Find(S, Index)
  else begin
    Index := 0;
    Result := False;
  end;
end;

procedure TTntWideStringListHelper.InheritedSort;
begin
  inherited Sort;
end;

procedure TTntWideStringListHelper.Sort;
begin
  if Parent <> nil then
    Parent.Sort;
end;

{ TTntWideStringList }

constructor TTntWideStringList.Create;
begin
  inherited Create;
  FList := TTntWideStringListHelper.Create;
  TTntWideStringListHelper(FList).Parent := Self;
end;

destructor TTntWideStringList.Destroy;
begin
  TTntWideStringListHelper(FList).Parent := nil;
  FreeAndNil(FList);
  inherited;
end;

function TTntWideStringList.GetDuplicates: TDuplicates;
begin
  Result := FList.Duplicates;
end;

procedure TTntWideStringList.SetDuplicates(const Value: TDuplicates);
begin
  FList.Duplicates := Value;
end;

function TTntWideStringList.GetSorted: Boolean;
begin
  Result := FList.Sorted;
end;

procedure TTntWideStringList.SetSorted(const Value: Boolean);
begin
  FList.Sorted := Value;
end;

function TTntWideStringList.GetOnChange: TNotifyEvent;
begin
  Result := FList.OnChange;
end;

procedure TTntWideStringList.SetOnChange(const Value: TNotifyEvent);
begin
  FList.OnChange := Value;
end;

function TTntWideStringList.GetOnChanging: TNotifyEvent;
begin
  Result := FList.OnChanging;
end;

procedure TTntWideStringList.SetOnChanging(const Value: TNotifyEvent);
begin
  FList.OnChanging := Value;
end;

function TTntWideStringList.Add(const S: WideString): Integer;
begin
  Result := FList.Add(S);
end;

procedure TTntWideStringList.Clear;
begin
  FList.Clear;
end;

procedure TTntWideStringList.Delete(Index: Integer);
begin
  FList.Delete(Index);
end;

procedure TTntWideStringList.Exchange(Index1, Index2: Integer);
begin
  FList.Exchange(Index1, Index2);
end;

function TTntWideStringList.Get(Index: Integer): WideString;
begin
  Result := FList.Strings[Index];
end;

function TTntWideStringList.GetCapacity: Integer;
begin
  Result := FList.Capacity;
end;

function TTntWideStringList.GetCount: Integer;
begin
  Result := FList.Count;
end;

function TTntWideStringList.GetObject(Index: Integer): TObject;
begin
  Result := FList.Objects[Index];
end;

function TTntWideStringList.IndexOf(const S: WideString): Integer;
begin
  Result := FList.IndexOf(S);
end;

procedure TTntWideStringList.Insert(Index: Integer; const S: WideString);
begin
  FList.Insert(Index, S);
end;

procedure TTntWideStringList.Put(Index: Integer; const S: WideString);
begin
  FList.Strings[Index] := S;
end;

procedure TTntWideStringList.PutObject(Index: Integer; AObject: TObject);
begin
  FList.Objects[Index] := AObject;
end;

procedure TTntWideStringList.SetCapacity(NewCapacity: Integer);
begin
  FList.Capacity := NewCapacity;
end;

procedure TTntWideStringList.SetLanguage(Value: LCID);
begin
  FList.Language := Value;
end;

procedure TTntWideStringList.SetUpdateState(Updating: Boolean);
begin
  TTntWideStringListHelper(FList).SetUpdateState(Updating);
end;

procedure TTntWideStringList.Sort;
begin
  TTntWideStringListHelper(FList).InheritedSort;
end;

procedure TTntWideStringList.Changed;
begin
  TTntWideStringListHelper(FList).InheritedChanged;
end;

procedure TTntWideStringList.Changing;
begin
  TTntWideStringListHelper(FList).InheritedChanging;
end;

function TTntWideStringList.Find(const S: WideString; var Index: Integer): Boolean;
begin
  Result := TTntWideStringListHelper(FList).InheritedFind(S, Index);
end;

initialization
  Win32PlatformIsUnicode := (Win32Platform = VER_PLATFORM_WIN32_NT);
  Win32PlatformIsXP := ((Win32MajorVersion = 5) and (Win32MinorVersion >= 1))
                    or  (Win32MajorVersion > 5);

end.
