
{*******************************************************}
{    The Delphi Unicode Controls Project                }
{                                                       }
{      http://home.ccci.org/wolbrink                    }
{                                                       }
{ Copyright (c) 2002, Troy Wolbrink (wolbrink@ccci.org) }
{                                                       }
{*******************************************************}

unit TntComCtrls;

interface

uses Classes, Controls, {$IFNDEF VER130} ListActns, {$ENDIF} ComCtrls, Messages, Windows,
     CommCtrl, {$IFDEF JCL} JclUnicode, {$ELSE} Unicode, {$ENDIF} Contnrs, TntControls, TntClasses;

type
  TTntCustomListView = class;
  TTntListItems = class;

{TNT-WARN TListColumn}
  TTntListColumn = class(TListColumn{TNT-ALLOW TListColumn})
  private
    FCaption: WideString;
    procedure SetInheritedCaption(const Value: AnsiString);
    function GetCaption: WideString;
    procedure SetCaption(const Value: WideString);
  protected
    procedure DefineProperties(Filer: TFiler); override;
  public
    procedure Assign(Source: TPersistent); override;
  published
    property Caption: WideString read GetCaption write SetCaption;
  end;

{TNT-WARN TListColumns}
  TTntListColumns = class(TListColumns{TNT-ALLOW TListColumns})
  private
    function GetItem(Index: Integer): TTntListColumn;
    procedure SetItem(Index: Integer; Value: TTntListColumn);
  public
    constructor Create(AOwner: TTntCustomListView);
    function Add: TTntListColumn;
    function Owner: TTntCustomListView;
    property Items[Index: Integer]: TTntListColumn read GetItem write SetItem; default;
  end;

{TNT-WARN TListItem}
  TTntListItem = class(TListItem{TNT-ALLOW TListItem})
  private
    FCaption: WideString;
    FSubItems: TTntWideStrings;
    procedure SetInheritedCaption(const Value: AnsiString);
    function GetCaption: WideString;
    procedure SetCaption(const Value: WideString);
    procedure SetSubItems(const Value: TTntWideStrings);
    function GetListView: TTntCustomListView;
    function GetTntOwner: TTntListItems;
  public
    constructor Create(AOwner: TListItems{TNT-ALLOW TListItems}); virtual;
    destructor Destroy; override;
    property Owner: TTntListItems read GetTntOwner;
    property ListView: TTntCustomListView read GetListView;
    procedure Assign(Source: TPersistent); override;
    property Caption: WideString read GetCaption write SetCaption;
    property SubItems: TTntWideStrings read FSubItems write SetSubItems;
  end;

{TNT-WARN TListItems}
  TTntListItems = class(TListItems{TNT-ALLOW TListItems})
  private
    function GetItem(Index: Integer): TTntListItem;
    procedure SetItem(Index: Integer; const Value: TTntListItem);
  public
    function Owner: TTntCustomListView;
    property Item[Index: Integer]: TTntListItem read GetItem write SetItem; default;
    function Add: TTntListItem;
{$IFNDEF VER130}
    function AddItem(Item: TTntListItem; Index: Integer = -1): TTntListItem;
{$ENDIF}
    function Insert(Index: Integer): TTntListItem;
  end;

  TTntLVEditedEvent = procedure(Sender: TObject; Item: TTntListItem; var S: WideString) of object;
  TTntLVOwnerDataFindEvent = procedure(Sender: TObject; Find: TItemFind;
    const FindString: WideString; const FindPosition: TPoint; FindData: Pointer;
    StartIndex: Integer; Direction: TSearchDirection; Wrap: Boolean;
    var Index: Integer) of object;

{TNT-WARN TCustomListView}
  TTntCustomListView = class(TCustomListView{TNT-ALLOW TCustomListView}, IWideCustomListControl)
  private
    FEditHandle: THandle;
    FEditInstance: Pointer;
    FDefEditProc: Pointer;
    FOnEdited: TTntLVEditedEvent;
    FOnDataFind: TTntLVOwnerDataFindEvent;
    procedure EditWndProcW(var Message: TMessage);
    procedure BeginChangingWideItem;
    procedure EndChangingWideItem;
    function GetHint: WideString;
    procedure SetHint(const Value: WideString);
    function IsHintStored: Boolean;
    function GetListColumns: TTntListColumns;
    procedure SetListColumns(const Value: TTntListColumns);
    function ColumnFromIndex(Index: Integer): TTntListColumn;
    function GetColumnFromTag(Tag: Integer): TTntListColumn;
  private
    FSavedItems: TObjectList;
    FTestingForSortProc: Boolean;
    PWideFindString: PWideChar;
    FChangingWideItemCount: Integer;
    OriginalDispInfoMask: Cardinal;
    CurrentDispInfo: PLVDispInfoW;
    FTempItem: TTntListItem;
    FListItems: TTntListItems;
    function AreItemsStored: Boolean;
    procedure SetItems(Value: TTntListItems);
    procedure CNNotify(var Message: TWMNotify); message CN_NOTIFY;
    function GetItemW(Value: TLVItemW): TTntListItem;
    procedure WMNotify(var Message: TWMNotify); message WM_NOTIFY;
  protected
    procedure CreateWindowHandle(const Params: TCreateParams); override;
    procedure DefineProperties(Filer: TFiler); override;
    function GetActionLinkClass: TControlActionLinkClass; override;
    procedure ActionChange(Sender: TObject; CheckDefaults: Boolean); override;
    procedure CreateWnd; override;
    procedure DestroyWnd; override;
    procedure WndProc(var Message: TMessage); override;
    function OwnerDataFetch(Item: TListItem{TNT-ALLOW TListItem}; Request: TItemRequest): Boolean; override;
    function CreateListItem: TListItem{TNT-ALLOW TListItem}; override;
    property Items: TTntListItems read FListItems write SetItems stored AreItemsStored;
    procedure Edit(const Item: TLVItem); override;
    function OwnerDataFind(Find: TItemFind; const FindString: AnsiString;
      const FindPosition: TPoint; FindData: Pointer; StartIndex: Integer;
      Direction: TSearchDirection; Wrap: Boolean): Integer; override;
    property Columns: TTntListColumns read GetListColumns write SetListColumns;
    procedure DrawItem(Item: TListItem{TNT-ALLOW TListItem}; Rect: TRect; State: TOwnerDrawState); override;
    property OnEdited: TTntLVEditedEvent read FOnEdited write FOnEdited;
    property OnDataFind: TTntLVOwnerDataFindEvent read FOnDataFind write FOnDataFind;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property Column[Index: Integer]: TTntListColumn read ColumnFromIndex;
{$IFNDEF VER130}
    procedure CopySelection(Destination: TCustomListControl); override;
{$ENDIF}
    procedure AddItem(const Item: WideString; AObject: TObject); reintroduce; virtual;
    function FindCaption(StartIndex: Integer; Value: WideString; Partial,
      Inclusive, Wrap: Boolean): TTntListItem;
    function GetSearchString: WideString;
    function StringWidth(S: WideString): Integer;
  published
    property Hint: WideString read GetHint write SetHint stored IsHintStored;
  end;

{TNT-WARN TListView}
  TTntListView = class(TTntCustomListView)
  published
    property Action;
    property Align;
    property AllocBy;
    property Anchors;
    property BevelEdges;
    property BevelInner;
    property BevelOuter;
    property BevelKind default bkNone;
    property BevelWidth;
    property BiDiMode;
    property BorderStyle;
    property BorderWidth;
    property Checkboxes;
    property Color;
    property Columns;
    property ColumnClick;
    property Constraints;
    property Ctl3D;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property Font;
    property FlatScrollBars;
    property FullDrag;
    property GridLines;
    property HideSelection;
    property HotTrack;
    property HotTrackStyles;
    property HoverTime;
    property IconOptions;
    property Items;
    property LargeImages;
    property MultiSelect;
    property OwnerData;
    property OwnerDraw;
    property ReadOnly default False;
    property RowSelect;
    property ParentBiDiMode;
    property ParentColor default False;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowColumnHeaders;
    property ShowWorkAreas;
    property ShowHint;
    property SmallImages;
    property SortType;
    property StateImages;
    property TabOrder;
    property TabStop default True;
    property ViewStyle;
    property Visible;
    property OnAdvancedCustomDraw;
    property OnAdvancedCustomDrawItem;
    property OnAdvancedCustomDrawSubItem;
    property OnChange;
    property OnChanging;
    property OnClick;
    property OnColumnClick;
    property OnColumnDragged;
    property OnColumnRightClick;
    property OnCompare;
    property OnContextPopup;
    property OnCustomDraw;
    property OnCustomDrawItem;
    property OnCustomDrawSubItem;
    property OnData;
    property OnDataFind;
    property OnDataHint;
    property OnDataStateChange;
    property OnDblClick;
    property OnDeletion;
    property OnDrawItem;
    property OnEdited;
    property OnEditing;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnGetImageIndex;
    property OnGetSubItemImage;
    property OnDragDrop;
    property OnDragOver;
    property OnInfoTip;
    property OnInsert;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnResize;
    property OnSelectItem;
    property OnStartDock;
    property OnStartDrag;
  end;

type
{TNT-WARN TCustomRichEdit}
  TTntCustomRichEdit = class(TCustomRichEdit{TNT-ALLOW TCustomRichEdit})
  private
    FRichEditStrings: TTntWideStrings;
    FPrintingTextLength: Integer;
    procedure WMGetTextLength(var Message: TWMGetTextLength); message WM_GETTEXTLENGTH;
    procedure SetRichEditStrings(const Value: TTntWideStrings);
    function GetWideSelText: WideString;
    function GetText: WideString;
    procedure SetWideSelText(const Value: WideString);
    procedure SetText(const Value: WideString);
    function GetHint: WideString;
    function IsHintStored: Boolean;
    procedure SetHint(const Value: WideString);
  protected
    procedure CreateParams(var Params: TCreateParams); override;
    procedure CreateWindowHandle(const Params: TCreateParams); override;
    procedure CreateWnd; override;
    procedure DefineProperties(Filer: TFiler); override;
    function GetActionLinkClass: TControlActionLinkClass; override;
    procedure ActionChange(Sender: TObject; CheckDefaults: Boolean); override;
    function GetSelText: string{TNT-ALLOW string}; override;
    function CharPosToGet(CharPos: Integer): Integer;
    function CharPosToSet(CharPos: Integer): Integer;
    function GetSelLength: Integer; override;
    function GetSelStart: Integer; override;
    procedure SetSelLength(Value: Integer); override;
    procedure SetSelStart(Value: Integer); override;
    function LineBreakStyle: TTntTextLineBreakStyle;
    property Lines: TTntWideStrings read FRichEditStrings write SetRichEditStrings;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Print(const Caption: string{TNT-ALLOW string}); override;
    property SelText: WideString read GetWideSelText write SetWideSelText;
    property Text: WideString read GetText write SetText;
  published
    property Hint: WideString read GetHint write SetHint stored IsHintStored;
  end;

{TNT-WARN TRichEdit}
  TTntRichEdit = class(TTntCustomRichEdit)
  published
    property Align;
    property Alignment;
    property Anchors;
    property BevelEdges;
    property BevelInner;
    property BevelOuter;
    property BevelKind default bkNone;
    property BevelWidth;
    property BiDiMode;
    property BorderStyle;
    property BorderWidth;
    property Color;
    property Ctl3D;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property Font;
    property HideSelection;
    property HideScrollBars;
    property ImeMode;
    property ImeName;
    property Constraints;
    property Lines;
    property MaxLength;
    property ParentBiDiMode;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PlainText;
    property PopupMenu;
    property ReadOnly;
    property ScrollBars;
    property ShowHint;
    property TabOrder;
    property TabStop default True;
    property Visible;
    property WantTabs;
    property WantReturns;
    property WordWrap;
    property OnChange;
    property OnContextPopup;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnMouseWheel;
    property OnMouseWheelDown;
    property OnMouseWheelUp;
    property OnProtectChange;
    property OnResizeRequest;
    property OnSaveClipboard;
    property OnSelectionChange;
    property OnStartDock;
    property OnStartDrag;
  end;

type
{TNT-WARN TCustomTabControl}
  TTntCustomTabControl = class(TCustomTabControl{TNT-ALLOW TCustomTabControl})
  private
    FTabs: TTntWideStrings;
    FSaveTabIndex: Integer;
    FSaveTabs: TTntWideStrings;
    procedure SetTabs(const Value: TTntWideStrings);
    procedure CMDialogChar(var Message: TCMDialogChar); message CM_DIALOGCHAR;
    function GetHint: WideString;
    function IsHintStored: Boolean;
    procedure SetHint(const Value: WideString);
  protected
    procedure CreateWindowHandle(const Params: TCreateParams); override;
    procedure DefineProperties(Filer: TFiler); override;
    function GetActionLinkClass: TControlActionLinkClass; override;
    procedure ActionChange(Sender: TObject; CheckDefaults: Boolean); override;
    procedure CreateWnd; override;
    procedure DestroyWnd; override;
    property Tabs: TTntWideStrings read FTabs write SetTabs;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property Hint: WideString read GetHint write SetHint stored IsHintStored;
  end;

{TNT-WARN TTabControl}
  TTntTabControl = class(TTntCustomTabControl)
  public
    property DisplayRect;
  published
    property Align;
    property Anchors;
    property BiDiMode;
    property Constraints;
    property DockSite;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property Font;
    property HotTrack;
    property Images;
    property MultiLine;
    property MultiSelect;
    property OwnerDraw;
    property ParentBiDiMode;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property RaggedRight;
    property ScrollOpposite;
    property ShowHint;
    property Style;
    property TabHeight;
    property TabOrder;
    property TabPosition;
    property Tabs;
    property TabIndex;  // must be after Tabs
    property TabStop;
    property TabWidth;
    property Visible;
    property OnChange;
    property OnChanging;
    property OnContextPopup;
    property OnDockDrop;
    property OnDockOver;
    property OnDragDrop;
    property OnDragOver;
    property OnDrawTab;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnGetImageIndex;
    property OnGetSiteInfo;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnResize;
    property OnStartDock;
    property OnStartDrag;
    property OnUnDock;
  end;

type
{TNT-WARN TTabSheet}
  TTntTabSheet = class(TTabSheet{TNT-ALLOW TTabSheet})
  private
    Force_Inherited_WMSETTEXT: Boolean;
    function IsCaptionStored: Boolean;
    function GetCaption: TWideCaption;
    procedure SetCaption(const Value: TWideCaption);
    procedure WMSetText(var Message: TWMSetText); message WM_SETTEXT;
    function GetHint: WideString;
    function IsHintStored: Boolean;
    procedure SetHint(const Value: WideString);
  protected
    procedure CreateWindowHandle(const Params: TCreateParams); override;
    procedure DefineProperties(Filer: TFiler); override;
    function GetActionLinkClass: TControlActionLinkClass; override;
    procedure ActionChange(Sender: TObject; CheckDefaults: Boolean); override;
  published
    property Caption: TWideCaption read GetCaption write SetCaption stored IsCaptionStored;
    property Hint: WideString read GetHint write SetHint stored IsHintStored;
  end;

{TNT-WARN TPageControl}
  TTntPageControl = class(TPageControl{TNT-ALLOW TPageControl})
  private
    FNewDockSheet: TTntTabSheet;
    function IsHintStored: Boolean;
    function GetHint: WideString;
    procedure SetHint(const Value: WideString);
    procedure CMDialogChar(var Message: TCMDialogChar); message CM_DIALOGCHAR;
    procedure CMDockNotification(var Message: TCMDockNotification); message CM_DOCKNOTIFICATION;
    procedure CMDockClient(var Message: TCMDockClient); message CM_DOCKCLIENT;
  protected
    procedure CreateWindowHandle(const Params: TCreateParams); override;
    procedure DefineProperties(Filer: TFiler); override;
    function GetActionLinkClass: TControlActionLinkClass; override;
    procedure ActionChange(Sender: TObject; CheckDefaults: Boolean); override;
    procedure WndProc(var Message: TMessage); override;
    procedure DoAddDockClient(Client: TControl; const ARect: TRect); override;
  published
    property Hint: WideString read GetHint write SetHint stored IsHintStored;
  end;

{TNT-WARN TTrackBar}
  TTntTrackBar = class(TTrackBar{TNT-ALLOW TTrackBar})
  private
    function IsHintStored: Boolean;
    function GetHint: WideString;
    procedure SetHint(const Value: WideString);
  protected
    procedure CreateWindowHandle(const Params: TCreateParams); override;
    procedure DefineProperties(Filer: TFiler); override;
    function GetActionLinkClass: TControlActionLinkClass; override;
    procedure ActionChange(Sender: TObject; CheckDefaults: Boolean); override;
  published
    property Hint: WideString read GetHint write SetHint stored IsHintStored;
  end;

{TNT-WARN TProgressBar}
  TTntProgressBar = class(TProgressBar{TNT-ALLOW TProgressBar})
  private
    function IsHintStored: Boolean;
    function GetHint: WideString;
    procedure SetHint(const Value: WideString);
  protected
    procedure CreateWindowHandle(const Params: TCreateParams); override;
    procedure DefineProperties(Filer: TFiler); override;
    function GetActionLinkClass: TControlActionLinkClass; override;
    procedure ActionChange(Sender: TObject; CheckDefaults: Boolean); override;
  published
    property Hint: WideString read GetHint write SetHint stored IsHintStored;
  end;

{TNT-WARN TCustomUpDown}
  TTntCustomUpDown = class(TCustomUpDown{TNT-ALLOW TCustomUpDown})
  private
    function IsHintStored: Boolean;
    function GetHint: WideString;
    procedure SetHint(const Value: WideString);
  protected
    procedure CreateWindowHandle(const Params: TCreateParams); override;
    procedure DefineProperties(Filer: TFiler); override;
    function GetActionLinkClass: TControlActionLinkClass; override;
    procedure ActionChange(Sender: TObject; CheckDefaults: Boolean); override;
  published
    property Hint: WideString read GetHint write SetHint stored IsHintStored;
  end;

{TNT-WARN TUpDown}
  TTntUpDown = class(TTntCustomUpDown)
  published
    property AlignButton;
    property Anchors;
    property Associate;
    property ArrowKeys;
    property Enabled;
    property Hint;
    property Min;
    property Max;
    property Increment;
    property Constraints;
    property Orientation;
    property ParentShowHint;
    property PopupMenu;
    property Position;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Thousands;
    property Visible;
    property Wrap;
    property OnChanging;
    property OnChangingEx;
    property OnContextPopup;
    property OnClick;
    property OnEnter;
    property OnExit;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

{TNT-WARN TDateTimePicker}
  TTntDateTimePicker = class(TDateTimePicker{TNT-ALLOW TDateTimePicker})
  private
    function IsHintStored: Boolean;
    function GetHint: WideString;
    procedure SetHint(const Value: WideString);
  protected
    procedure CreateWindowHandle(const Params: TCreateParams); override;
    procedure DefineProperties(Filer: TFiler); override;
    function GetActionLinkClass: TControlActionLinkClass; override;
    procedure ActionChange(Sender: TObject; CheckDefaults: Boolean); override;
  published
    property Hint: WideString read GetHint write SetHint stored IsHintStored;
  end;

{TNT-WARN TMonthCalendar}
  TTntMonthCalendar = class(TMonthCalendar{TNT-ALLOW TMonthCalendar})
  private
    function IsHintStored: Boolean;
    function GetHint: WideString;
    procedure SetHint(const Value: WideString);
    function GetDate: TDate;
    procedure SetDate(const Value: TDate);
  protected
    procedure CreateWindowHandle(const Params: TCreateParams); override;
    procedure DefineProperties(Filer: TFiler); override;
    function GetActionLinkClass: TControlActionLinkClass; override;
    procedure ActionChange(Sender: TObject; CheckDefaults: Boolean); override;
  public
    procedure ForceGetMonthInfo;
  published
    property Date: TDate read GetDate write SetDate;
    property Hint: WideString read GetHint write SetHint stored IsHintStored;
  end;

{TNT-WARN TPageScroller}
  TTntPageScroller = class(TPageScroller{TNT-ALLOW TPageScroller})
  private
    function IsHintStored: Boolean;
    function GetHint: WideString;
    procedure SetHint(const Value: WideString);
  protected
    procedure CreateWindowHandle(const Params: TCreateParams); override;
    procedure DefineProperties(Filer: TFiler); override;
    function GetActionLinkClass: TControlActionLinkClass; override;
    procedure ActionChange(Sender: TObject; CheckDefaults: Boolean); override;
  published
    property Hint: WideString read GetHint write SetHint stored IsHintStored;
  end;

implementation

uses SysUtils, Forms, TntGraphics, Graphics, ImgList, TntStdCtrls, StdCtrls,
  RichEdit, TntWideStrPropHelper, ActiveIMM_TLB, Printers, TntForms, ComStrs, TntActnList;

procedure CreateUnicodeHandle_ComCtl(Control: TWinControl; const Params: TCreateParams;
  const SubClass: WideString);
begin
  Assert(SubClass <> '', 'TNT Internal Error: Only call CreateUnicodeHandle_ComCtl for Common Controls.');
  CreateUnicodeHandle(Control, Params, SubClass);
  if Win32PlatformIsUnicode then
    SendMessageW(Control.Handle, CCM_SETUNICODEFORMAT, Integer(True), 0);
end;

{ TTntListColumn }

procedure TTntListColumn.Assign(Source: TPersistent);
begin
  inherited;
  if Source is TTntListColumn then
    Caption := TTntListColumn(Source).Caption
  else if Source is TListColumn{TNT-ALLOW TListColumn} then
    FCaption := TListColumn{TNT-ALLOW TListColumn}(Source).Caption;
end;

procedure TTntListColumn.DefineProperties(Filer: TFiler);
begin
  inherited;
  DefineWideProperties(Filer, Self);
end;

procedure TTntListColumn.SetInheritedCaption(const Value: AnsiString);
begin
  inherited Caption := Value;
end;

function TTntListColumn.GetCaption: WideString;
begin
  Result := GetSyncedWideString(FCaption, inherited Caption);
end;

procedure TTntListColumn.SetCaption(const Value: WideString);
begin
  SetSyncedWideString(Value, FCaption, inherited Caption, SetInheritedCaption);
end;

{ TTntListColumns }

{$IFDEF VER130} // Delphi 5
type
  THackCollection = class(TPersistent)
  protected
    FItemClass: TCollectionItemClass;
  end;
{$ENDIF}
{$IFDEF VER140} // Delphi 6
type
  THackCollection = class(TPersistent)
  protected
    FItemClass: TCollectionItemClass;
  end;
{$ENDIF}
{$IFDEF VER150} // Delphi 7
type
  THackCollection = class(TPersistent)
  protected
    FItemClass: TCollectionItemClass;
  end;
{$ENDIF}

constructor TTntListColumns.Create(AOwner: TTntCustomListView);
begin
  inherited Create(AOwner);
  Assert(THackCollection(Self).FItemClass = Self.ItemClass, 'Internal Error in TTntListColumns.Create().');
  THackCollection(Self).FItemClass := TTntListColumn
end;

function TTntListColumns.Owner: TTntCustomListView;
begin
  Result := inherited Owner as TTntCustomListView;
end;

function TTntListColumns.Add: TTntListColumn;
begin
  Result := (inherited Add) as TTntListColumn;
end;

function TTntListColumns.GetItem(Index: Integer): TTntListColumn;
begin
  Result := inherited Items[Index] as TTntListColumn;
end;

procedure TTntListColumns.SetItem(Index: Integer; Value: TTntListColumn);
begin
  inherited SetItem(Index, Value);
end;

{ TWideSubItems }
type
  TWideSubItems = class(TTntWideStringList)
  private
    FIgnoreInherited: Boolean;
    FInheritedOwner: TListItem{TNT-ALLOW TListItem};
    FOwner: TTntListItem;
  protected
    procedure Put(Index: Integer; const S: WideString); override;
    function GetObject(Index: Integer): TObject; override;
    procedure PutObject(Index: Integer; AObject: TObject); override;
    procedure SetUpdateState(Updating: Boolean); override;
  public
    procedure Insert(Index: Integer; const S: WideString); override;
    function Add(const S: WideString): Integer; override;
    function AddObject(const S: WideString; AObject: TObject): Integer; override;
    procedure Clear; override;
    procedure Delete(Index: Integer); override;
  public
    constructor Create(AOwner: TTntListItem);
  end;

constructor TWideSubItems.Create(AOwner: TTntListItem);
begin
  inherited Create;
  FInheritedOwner := AOwner;
  FOwner := AOwner;
end;

function TWideSubItems.Add(const S: WideString): Integer;
begin
  FOwner.ListView.BeginChangingWideItem;
  try
    Result := inherited Add(S);
    if (not FIgnoreInherited) then
      FInheritedOwner.SubItems.Add(S);
  finally
    FOwner.ListView.EndChangingWideItem;
  end;
end;

function TWideSubItems.AddObject(const S: WideString; AObject: TObject): Integer;
begin
  FOwner.ListView.BeginChangingWideItem;
  try
    Result := inherited AddObject(S, AObject);
    if (not FIgnoreInherited) then
      FInheritedOwner.SubItems.AddObject(S, AObject);
  finally
    FOwner.ListView.EndChangingWideItem;
  end;
end;

procedure TWideSubItems.Clear;
begin
  FOwner.ListView.BeginChangingWideItem;
  try
    inherited;
    if (not FIgnoreInherited) then
      FInheritedOwner.SubItems.Clear;
  finally
    FOwner.ListView.EndChangingWideItem;
  end;
end;

procedure TWideSubItems.Delete(Index: Integer);
begin
  FOwner.ListView.BeginChangingWideItem;
  try
    inherited;
    if (not FIgnoreInherited) then
      FInheritedOwner.SubItems.Delete(Index);
  finally
    FOwner.ListView.EndChangingWideItem;
  end;
end;

procedure TWideSubItems.Insert(Index: Integer; const S: WideString);
begin
  FOwner.ListView.BeginChangingWideItem;
  try
    inherited;
    if (not FIgnoreInherited) then
      FInheritedOwner.SubItems.Insert(Index, S);
  finally
    FOwner.ListView.EndChangingWideItem;
  end;
end;

procedure TWideSubItems.Put(Index: Integer; const S: WideString);
begin
  FOwner.ListView.BeginChangingWideItem;
  try
    inherited;
    if (not FIgnoreInherited) then
      FInheritedOwner.SubItems[Index] := S;
  finally
    FOwner.ListView.EndChangingWideItem;
  end;
end;

function TWideSubItems.GetObject(Index: Integer): TObject;
begin
  Result := FInheritedOwner.SubItems.Objects[Index];
end;

procedure TWideSubItems.PutObject(Index: Integer; AObject: TObject);
begin
  FInheritedOwner.SubItems.Objects[Index] := AObject;
end;

type TAccessStrings = class(TStrings{TNT-ALLOW TStrings});

procedure TWideSubItems.SetUpdateState(Updating: Boolean);
begin
  inherited;
  TAccessStrings(FInheritedOwner.SubItems).SetUpdateState(Updating);
end;

{ TTntListItem }

constructor TTntListItem.Create(AOwner: TListItems{TNT-ALLOW TListItems});
begin
  inherited Create(AOwner);
  FSubItems := TWideSubItems.Create(Self);
end;

destructor TTntListItem.Destroy;
begin
  inherited;
  FreeAndNil(FSubItems);
end;

function TTntListItem.GetCaption: WideString;
begin
  Result := GetSyncedWideString(FCaption, inherited Caption);
end;

procedure TTntListItem.SetInheritedCaption(const Value: AnsiString);
begin
  inherited Caption := Value;
end;

procedure TTntListItem.SetCaption(const Value: WideString);
begin
  ListView.BeginChangingWideItem;
  try
    SetSyncedWideString(Value, FCaption, inherited Caption, SetInheritedCaption);
  finally
    ListView.EndChangingWideItem;
  end;
end;

procedure TTntListItem.Assign(Source: TPersistent);
begin
  if Source is TTntListItem then
    with Source as TTntListItem do
    begin
      Self.Caption := Caption;
      Self.Data := Data;
      Self.ImageIndex := ImageIndex;
      Self.Indent := Indent;
      Self.OverlayIndex := OverlayIndex;
      Self.StateIndex := StateIndex;
      Self.SubItems := SubItems;
      Self.Checked := Checked;
    end
  else inherited Assign(Source);
end;

procedure TTntListItem.SetSubItems(const Value: TTntWideStrings);
begin
  if Value <> nil then
    FSubItems.Assign(Value);
end;

function TTntListItem.GetTntOwner: TTntListItems;
begin
  Result := ListView.Items;
end;

function TTntListItem.GetListView: TTntCustomListView;
begin
  Result := ((inherited Owner).Owner as TTntCustomListView);
end;

{ TTntListItems }

function TTntListItems.Add: TTntListItem;
begin
  Result := (inherited Add) as TTntListItem;
end;

{$IFNDEF VER130}
function TTntListItems.AddItem(Item: TTntListItem; Index: Integer): TTntListItem;
begin
  Result := (inherited AddItem(Item, Index)) as TTntListItem;
end;
{$ENDIF}

function TTntListItems.Insert(Index: Integer): TTntListItem;
begin
  Result := (inherited Insert(Index)) as TTntListItem;
end;

function TTntListItems.GetItem(Index: Integer): TTntListItem;
begin
  Result := (inherited Item[Index]) as TTntListItem;
end;

function TTntListItems.Owner: TTntCustomListView;
begin
  Result := (inherited Owner) as TTntCustomListView;
end;

procedure TTntListItems.SetItem(Index: Integer; const Value: TTntListItem);
begin
  inherited Item[Index] := Value;
end;

{ TSavedListItem }
type
  TSavedListItem = class
    FCaption: WideString;
    FSubItems: TTntWideStrings;
    constructor Create;
    destructor Destroy; override;
  end;

constructor TSavedListItem.Create;
begin
  inherited;
  FSubItems := TTntWideStringList.Create;
end;

destructor TSavedListItem.Destroy;
begin
  FSubItems.Free;
  inherited;
end;

{ TTntCustomListView }

{$IFDEF VER130} // Delphi 5
type
  THackCustomListView = class(TWinControl)
  protected
    FxxxCanvas: TCanvas;
    FxxxBorderStyle: TBorderStyle;
    FxxxViewStyle: TViewStyle;
    FxxxReadOnly: Boolean;
    FxxxLargeImages: TCustomImageList;
    FxxxSmallImages: TCustomImageList;
    FxxxStateImages: TCustomImageList;
    FxxxDragImage: TDragImageList;
    FxxxMultiSelect: Boolean;
    FxxxSortType: TSortType;
    FxxxColumnClick: Boolean;
    FxxxShowColumnHeaders: Boolean;
    FxxxListItems: TListItems{TNT-ALLOW TListItems};
    FxxxClicked: Boolean;
    FxxxRClicked: Boolean;
    FxxxIconOptions: TIconOptions;
    FxxxHideSelection: Boolean;
    FListColumns: TListColumns{TNT-ALLOW TListColumns};
  end;
{$ENDIF}
{$IFDEF VER140} // Delphi 6
type
  THackCustomListView = class(TCustomMultiSelectListControl)
  protected
    FxxxCanvas: TCanvas;
    FxxxBorderStyle: TBorderStyle;
    FxxxViewStyle: TViewStyle;
    FxxxReadOnly: Boolean;
    FxxxLargeImages: TCustomImageList;
    FxxxSmallImages: TCustomImageList;
    FxxxStateImages: TCustomImageList;
    FxxxDragImage: TDragImageList;
    FxxxMultiSelect: Boolean;
    FxxxSortType: TSortType;
    FxxxColumnClick: Boolean;
    FxxxShowColumnHeaders: Boolean;
    FxxxListItems: TListItems{TNT-ALLOW TListItems};
    FxxxClicked: Boolean;
    FxxxRClicked: Boolean;
    FxxxIconOptions: TIconOptions;
    FxxxHideSelection: Boolean;
    FListColumns: TListColumns{TNT-ALLOW TListColumns};
  end;
{$ENDIF}
{$IFDEF VER150} // Delphi 7
type
  THackCustomListView = class(TCustomMultiSelectListControl)
  protected
    FxxxCanvas: TCanvas;
    FxxxBorderStyle: TBorderStyle;
    FxxxViewStyle: TViewStyle;
    FxxxReadOnly: Boolean;
    FxxxLargeImages: TCustomImageList;
    FxxxSmallImages: TCustomImageList;
    FxxxStateImages: TCustomImageList;
    FxxxDragImage: TDragImageList;
    FxxxMultiSelect: Boolean;
    FxxxSortType: TSortType;
    FxxxColumnClick: Boolean;
    FxxxShowColumnHeaders: Boolean;
    FxxxListItems: TListItems{TNT-ALLOW TListItems};
    FxxxClicked: Boolean;
    FxxxRClicked: Boolean;
    FxxxIconOptions: TIconOptions;
    FxxxHideSelection: Boolean;
    FListColumns: TListColumns{TNT-ALLOW TListColumns};
  end;
{$ENDIF}

var
  ComCtrls_DefaultListViewSort: TLVCompare = nil;

constructor TTntCustomListView.Create(AOwner: TComponent);
begin
  inherited;
  FListItems := TTntListItems.Create(Self);
  FEditInstance := TntClasses.MakeObjectInstance(EditWndProcW);
  Assert(THackCustomListView(Self).FListColumns = inherited Columns, 'Internal Error in TTntCustomListView.Create().');
  FreeAndNil(THackCustomListView(Self).FListColumns);
  THackCustomListView(Self).FListColumns := TTntListColumns.Create(Self);
end;

destructor TTntCustomListView.Destroy;
begin
  inherited;
  FreeAndNil(FListItems);
  TntClasses.FreeObjectInstance(FEditInstance);
end;

procedure TTntCustomListView.CreateWindowHandle(const Params: TCreateParams);

  procedure Capture_ComCtrls_DefaultListViewSort;
  begin
    FTestingForSortProc := True;
    try
      AlphaSort;
    finally
      FTestingForSortProc := False;
    end;
  end;
var
  Column: TLVColumn;
begin
  CreateUnicodeHandle_ComCtl(Self, Params, WC_LISTVIEW);
  if (Win32PlatformIsUnicode) then begin
    if not Assigned(ComCtrls_DefaultListViewSort) then
      Capture_ComCtrls_DefaultListViewSort;
    // the only way I could get editing to work is after a column had been inserted
    Column.mask := 0;
    ListView_InsertColumn(Handle, 0, Column);
    ListView_DeleteColumn(Handle, 0);
  end;
end;

procedure TTntCustomListView.DefineProperties(Filer: TFiler);
begin
  inherited;
  DefineWideProperties(Filer, Self);
end;

procedure TTntCustomListView.CreateWnd;
begin
  inherited;
  FreeAndNil(FSavedItems);
end;

procedure TTntCustomListView.DestroyWnd;
var
  i: integer;
  FSavedItem: TSavedListItem;
  Item: TTntListItem;
begin
  if (not (csDestroying in ComponentState)) and (not OwnerData) then begin
    FSavedItems := TObjectList.Create(True);
    for i := 0 to Items.Count - 1 do begin
      FSavedItem := TSavedListItem.Create;
      Item := Items[i];
      FSavedItem.FCaption := Item.FCaption;
      FSavedItem.FSubItems.Assign(Item.FSubItems);
      FSavedItems.Add(FSavedItem)
    end;
  end;
  inherited;
end;

function TTntCustomListView.GetListColumns: TTntListColumns;
begin
  Result := inherited Columns as TTntListColumns;
end;

procedure TTntCustomListView.SetListColumns(const Value: TTntListColumns);
begin
  Columns := Value;
end;

{$IFDEF VER130} // Delphi 5
type
  THackListColumn = class(TCollectionItem)
  protected
    FxxxAlignment: TAlignment;
    FxxxAutoSize: Boolean;
    FxxxCaption: AnsiString;
    FxxxMaxWidth: TWidth;
    FxxxMinWidth: TWidth;
    FxxxImageIndex: TImageIndex;
    FxxxPrivateWidth: TWidth;
    FxxxWidth: TWidth;
    FOrderTag: Integer;
  end;
{$ENDIF}
{$IFDEF VER140} // Delphi 6
type
  THackListColumn = class(TCollectionItem)
  protected
    FxxxAlignment: TAlignment;
    FxxxAutoSize: Boolean;
    FxxxCaption: AnsiString;
    FxxxMaxWidth: TWidth;
    FxxxMinWidth: TWidth;
    FxxxImageIndex: TImageIndex;
    FxxxPrivateWidth: TWidth;
    FxxxWidth: TWidth;
    FOrderTag: Integer;
  end;
{$ENDIF}
{$IFDEF VER150} // Delphi 7
type
  THackListColumn = class(TCollectionItem)
  protected
    FxxxAlignment: TAlignment;
    FxxxAutoSize: Boolean;
    FxxxCaption: AnsiString;
    FxxxMaxWidth: TWidth;
    FxxxMinWidth: TWidth;
    FxxxImageIndex: TImageIndex;
    FxxxPrivateWidth: TWidth;
    FxxxWidth: TWidth;
    FOrderTag: Integer;
  end;
{$ENDIF}

function TTntCustomListView.GetColumnFromTag(Tag: Integer): TTntListColumn;
var
  I: Integer;
begin
  for I := 0 to Columns.Count - 1 do
  begin
    Result := Columns[I];
    if THackListColumn(Result).FOrderTag = Tag then Exit;
  end;
  Result := nil;
end;

function TTntCustomListView.ColumnFromIndex(Index: Integer): TTntListColumn;
begin
  Result := inherited Column[Index] as TTntListColumn;
end;

function TTntCustomListView.AreItemsStored: Boolean;
begin
{$IFDEF VER130}
  Result := not OwnerData;
{$ELSE}
  if Assigned(Action) then
  begin
    if Action is TCustomListAction{TNT-ALLOW TCustomListAction} then
      Result := False
    else
      Result := True;
  end
  else
    Result := not OwnerData;
{$ENDIF}
end;

procedure TTntCustomListView.SetItems(Value: TTntListItems);
begin
  FListItems.Assign(Value);
end;

type TTntListItemClass = class of TTntListItem;

function TTntCustomListView.CreateListItem: TListItem{TNT-ALLOW TListItem};
var
  LClass: TClass;
  TntLClass: TTntListItemClass;
begin
  LClass := TTntListItem;
{$IFNDEF VER130}
  if Assigned(OnCreateItemClass) then
    OnCreateItemClass(Self, TListItemClass(LClass));
  if not LClass.InheritsFrom(TTntListItem) then
    raise Exception.Create('Internal Error: OnCreateItemClass.ItemClass must inherit from TTntListItem.');
{$ENDIF}
  TntLClass := TTntListItemClass(LClass);
  Result := TntLClass.Create(inherited Items);
  if FTempItem = nil then
    FTempItem := Result as TTntListItem; { In Delphi 5/6, the first item creates is the temp item }
end;

function TTntCustomListView.GetItemW(Value: TLVItemW): TTntListItem;
begin
  with Value do begin
    if OwnerData then
      Result := FTempItem
    else if (mask and LVIF_PARAM) <> 0 then
      Result := TListItem{TNT-ALLOW TListItem}(lParam) as TTntListItem
    else if iItem >= 0 then
      Result := Items[IItem]
    else
      Result := nil
  end;
end;

function TTntCustomListView.OwnerDataFetch(Item: TListItem{TNT-ALLOW TListItem}; Request: TItemRequest): Boolean;
begin
  if  (CurrentDispInfo <> nil)
  and (OriginalDispInfoMask and LVIF_TEXT <> 0) then begin
    (Item as TTntListItem).FCaption := CurrentDispInfo.item.pszText
  end;
  (Item as TTntListItem).FSubItems.Clear;
  Result := inherited OwnerDataFetch(Item, Request);
end;

function TntDefaultListViewSort(Item1, Item2: TTntListItem; lParam: Integer): Integer; stdcall;
begin
  Assert(Win32PlatformIsUnicode);
  with Item1 do
    if Assigned(ListView.OnCompare) then
      ListView.OnCompare(ListView, Item1, Item2, lParam, Result)
    else Result := lstrcmpw(PWideChar(Item1.Caption), PWideChar(Item2.Caption));
end;

procedure TTntCustomListView.WndProc(var Message: TMessage);
var
  Item: TTntListItem;
  InheritedItem: TListItem{TNT-ALLOW TListItem};
  SubItem: Integer;
  SavedItem: TSavedListItem;
  PCol: PLVColumn;
  Col: TTntListColumn;
begin
  with Message do begin
    // restore previous values (during CreateWnd)
    if (FSavedItems <> nil) and (FChangingWideItemCount = 0) and (Msg = LVM_SETITEMTEXTA) then begin
      Item := Items[wParam];
      SavedItem := TSavedListItem(FSavedItems[wParam]);
      if (Msg = LVM_SETITEMTEXTA) and (PLVItem(lParam).iSubItem = 0) then
        Item.FCaption := SavedItem.FCaption
      else begin
        SubItem := PLVItem(lParam).iSubItem - 1;
        TWideSubItems(Item.SubItems).FIgnoreInherited := True;
        try
          if SubItem < Item.SubItems.Count then begin
            Item.SubItems[SubItem] := SavedItem.FSubItems[SubItem];
            Item.SubItems.Objects[SubItem] := SavedItem.FSubItems.Objects[SubItem]
          end else if SubItem = Item.SubItems.Count then
            Item.SubItems.AddObject(SavedItem.FSubItems[SubItem], SavedItem.FSubItems.Objects[SubItem])
          else
            Item.SubItems.Assign(SavedItem.FSubItems)
        finally
          TWideSubItems(Item.SubItems).FIgnoreInherited := False;
        end;
      end;
    end;

    // sync wide with ansi
    if (FSavedItems = nil) and (FChangingWideItemCount = 0) and (Msg = LVM_UPDATE) then begin
      Item := Items[wParam];
      InheritedItem := Item;
      TWideSubItems(Item.SubItems).FIgnoreInherited := True;
      try
        Item.SubItems.Assign(InheritedItem.SubItems)
      finally
        TWideSubItems(Item.SubItems).FIgnoreInherited := False;
      end;
    end;

    if (FSavedItems = nil) and (FChangingWideItemCount = 0) and (Msg = LVM_SETITEMTEXTA) then begin
      if OwnerData then
        Item := FTempItem
      else
        Item := Items[wParam];
      InheritedItem := Item;
      if (Msg = LVM_SETITEMTEXTA) and (PLVItem(lParam).iSubItem = 0) then
        Item.FCaption := InheritedItem.Caption
      else begin
        SubItem := PLVItem(lParam).iSubItem - 1;
        TWideSubItems(Item.SubItems).FIgnoreInherited := True;
        try
          if SubItem < Item.SubItems.Count then begin
            Item.SubItems[SubItem] := InheritedItem.SubItems[SubItem];
            Item.SubItems.Objects[SubItem] := InheritedItem.SubItems.Objects[SubItem]
          end else if SubItem = Item.SubItems.Count then
            Item.SubItems.AddObject(InheritedItem.SubItems[SubItem], InheritedItem.SubItems.Objects[SubItem])
          else
            Item.SubItems.Assign(InheritedItem.SubItems)
        finally
          TWideSubItems(Item.SubItems).FIgnoreInherited := False;
        end;
      end;
    end;

    // capture ANSI version of DefaultListViewSort from ComCtrls
    if (FTestingForSortProc)
    and (Msg = LVM_SORTITEMS) then begin
      ComCtrls_DefaultListViewSort := Pointer(lParam);
      exit;
    end;

    if (Msg = LVM_SETCOLUMNA) then begin
      // make sure that wide column caption stays in sync with ANSI
      PCol := PLVColumn(lParam);
      if (PCol.mask and LVCF_TEXT) <> 0 then begin
        Col := GetColumnFromTag(wParam);
        if (Col <> nil) and (AnsiString(Col.Caption) <> PCol.pszText) then begin
          Col.FCaption := PCol.pszText;
        end;
      end;
    end;

    if (Win32PlatformIsUnicode)
    and (Msg = LVM_SETITEMTEXTA) and (PLVItem(lParam).pszText = LPSTR_TEXTCALLBACK) then
      // Unicode:: call wide version of text call back instead
      Result := SendMessageW(Handle, LVM_SETITEMTEXTW, WParam, LParam)
    else if (Win32PlatformIsUnicode)
    and (Msg = LVM_SORTITEMS) and (Pointer(lParam) = @ComCtrls_DefaultListViewSort) then
      // Unicode:: call wide version of sort proc instread
      Result := SendMessageW(Handle, LVM_SORTITEMS, wParam, Integer(@TntDefaultListViewSort))
    else if (Win32PlatformIsUnicode)
    and (Msg = LVM_SETCOLUMNA) and ((PLVColumn(lParam).mask and LVCF_TEXT) <> 0)
    and (GetColumnFromTag(wParam) <> nil) then begin
      PLVColumn(lParam).pszText := PAnsiChar(PWideChar(GetColumnFromTag(wParam).FCaption));
      Result := SendMessageW(Handle, LVM_SETCOLUMNW, wParam, lParam);
    end else begin
      if (Msg = LVM_SETEXTENDEDLISTVIEWSTYLE) and CheckBoxes then begin
        { fix a bug in TCustomListView.ResetExStyles }
        lParam := lParam or LVS_EX_SUBITEMIMAGES or LVS_EX_INFOTIP;
      end;
      inherited;
    end;
  end;
end;

procedure TTntCustomListView.WMNotify(var Message: TWMNotify);
begin
  inherited;
  // capture updated info after inherited
  with Message.NMHdr^ do
    case code of
      HDN_ENDTRACKW:
        begin
          Message.NMHdr^.code := HDN_ENDTRACKA;
          try
            inherited
          finally
            Message.NMHdr^.code := HDN_ENDTRACKW;
          end;
        end;
      HDN_DIVIDERDBLCLICKW:
        begin
          Message.NMHdr^.code := HDN_DIVIDERDBLCLICKA;
          try
            inherited
          finally
            Message.NMHdr^.code := HDN_DIVIDERDBLCLICKW;
          end;
        end;
    end;
end;

procedure TTntCustomListView.CNNotify(var Message: TWMNotify);
var
  Item: TTntListItem;
begin
  if (not Win32PlatformIsUnicode) then
    inherited
  else begin
    with Message do
    begin
      case NMHdr^.code of
        HDN_TRACKW:
          begin
            NMHdr^.code := HDN_TRACKA;
            try
              inherited;
            finally
              NMHdr^.code := HDN_TRACKW;
            end;
          end;
        LVN_GETDISPINFOW:
          begin
            // call inherited without the LVIF_TEXT flag
            CurrentDispInfo := PLVDispInfoW(NMHdr);
            try
              OriginalDispInfoMask := PLVDispInfoW(NMHdr)^.item.mask;

              PLVDispInfoW(NMHdr)^.item.mask := PLVDispInfoW(NMHdr)^.item.mask and (not LVIF_TEXT);
              try
                NMHdr^.code := LVN_GETDISPINFOA;
                try
                  inherited;
                finally
                  NMHdr^.code := LVN_GETDISPINFOW;
                end;
              finally
                PLVDispInfoW(NMHdr)^.item.mask := OriginalDispInfoMask;
              end;
            finally
              CurrentDispInfo := nil;
            end;

            // handle any text info
            with PLVDispInfoW(NMHdr)^.item do
            begin
              if (mask and LVIF_TEXT) <> 0 then
              begin
                Item := GetItemW(PLVDispInfoW(NMHdr)^.item);
                if iSubItem = 0 then
                  StrLCopyW(pszText, PWideChar(Item.Caption), cchTextMax - 1)
                else begin
                  with Item.SubItems do begin
                    if iSubItem <= Count then
                      StrLCopyW(pszText, PWideChar(Strings[iSubItem - 1]), cchTextMax - 1)
                    else pszText[0] := #0;
                  end;
                end;
              end;
            end;
          end;
        LVN_ODFINDITEMW:
          with PNMLVFindItem(NMHdr)^ do
          begin
            if ((lvfi.flags and LVFI_PARTIAL) <> 0) or ((lvfi.flags and LVFI_STRING) <> 0) then
              PWideFindString := TLVFindInfoW(lvfi).psz
            else
              PWideFindString := nil;
            lvfi.psz := nil;
            NMHdr^.code := LVN_ODFINDITEMA;
            try
              inherited; {will Result in call to OwnerDataFind}
            finally
              TLVFindInfoW(lvfi).psz := PWideFindString;
              NMHdr^.code := LVN_ODFINDITEMW;
              PWideFindString := nil;
            end;
          end;
        LVN_BEGINLABELEDITW:
          begin
            Item := GetItemW(PLVDispInfoW(NMHdr)^.item);
            if not CanEdit(Item) then Result := 1;
            if Result = 0 then
            begin
              FEditHandle := ListView_GetEditControl(Handle);
              FDefEditProc := Pointer(GetWindowLong(FEditHandle, GWL_WNDPROC));
              SetWindowLongW(FEditHandle, GWL_WNDPROC, LongInt(FEditInstance));
            end;
          end;
        LVN_ENDLABELEDITW:
          with PLVDispInfoW(NMHdr)^ do
            if (item.pszText <> nil) and (item.IItem <> -1) then
              Edit(TLVItemA(item));
        LVN_GETINFOTIPW:
          begin
            NMHdr^.code := LVN_GETINFOTIPA;
            try
              inherited;
            finally
              NMHdr^.code := LVN_GETINFOTIPW;
            end;
          end;
        else
          inherited;
      end;
    end;
  end;
end;

function TTntCustomListView.OwnerDataFind(Find: TItemFind; const FindString: AnsiString;
  const FindPosition: TPoint; FindData: Pointer; StartIndex: Integer;
  Direction: TSearchDirection; Wrap: Boolean): Integer;
var
  WideFindString: WideString;
  AnsiEvent: TLVOwnerDataFindEvent;
begin
  if Assigned(PWideFindString) then
    WideFindString := PWideFindString
  else
    WideFindString := FindString;
  Result := -1;
  if Assigned(OnDataFind) then
    OnDataFind(Self, Find, WideFindString, FindPosition, FindData, StartIndex, Direction, Wrap, Result)
  else if Assigned(inherited OnDataFind) then begin
    AnsiEvent := inherited OnDataFind;
    AnsiEvent(Self, Find, WideFindString, FindPosition, FindData, StartIndex, Direction,
      Wrap, Result);
  end;
end;

procedure TTntCustomListView.Edit(const Item: TLVItem);
var
  S: WideString;
  AnsiS: AnsiString;
  EditItem: TTntListItem;
  AnsiEvent: TLVEditedEvent;
begin
  if (not Win32PlatformIsUnicode) then
    S := Item.pszText
  else
    S := TLVItemW(Item).pszText;
  EditItem := GetItemW(TLVItemW(Item));
  if Assigned(OnEdited) then
    OnEdited(Self, EditItem, S)
  else if Assigned(inherited OnEdited) then
  begin
    AnsiEvent := inherited OnEdited;
    AnsiS := S;
    AnsiEvent(Self, EditItem, AnsiS);
    S := AnsiS;
  end;
  if EditItem <> nil then
    EditItem.Caption := S;
end;

procedure TTntCustomListView.EditWndProcW(var Message: TMessage);
begin
  Assert(Win32PlatformIsUnicode);
  try
    with Message do
    begin
      case Msg of
        WM_KEYDOWN,
        WM_SYSKEYDOWN: if DoKeyDown(TWMKey(Message)) then Exit;
        WM_CHAR:
          begin
            MakeWMCharMsgSafeForAnsi(Message);
            try
              if DoKeyPress(TWMKey(Message)) then Exit;
            finally
              RestoreWMCharMsg(Message);
            end;
          end;
        WM_KEYUP,
        WM_SYSKEYUP: if DoKeyUp(TWMKey(Message)) then Exit;
        CN_KEYDOWN,
        CN_CHAR, CN_SYSKEYDOWN,
        CN_SYSCHAR:
          begin
            WndProc(Message);
            Exit;
          end;
      end;
      Result := CallWindowProcW(FDefEditProc, FEditHandle, Msg, WParam, LParam);
    end;
  except
    Application.HandleException(Self);
  end;
end;

procedure TTntCustomListView.BeginChangingWideItem;
begin
  Inc(FChangingWideItemCount);
end;

procedure TTntCustomListView.EndChangingWideItem;
begin
  if FChangingWideItemCount > 0 then
    Dec(FChangingWideItemCount);
end;

procedure TTntCustomListView.DrawItem(Item: TListItem{TNT-ALLOW TListItem}; Rect: TRect;
  State: TOwnerDrawState);
begin
  TControlCanvas(Canvas).UpdateTextFlags;
  if Assigned(OnDrawItem) then OnDrawItem(Self, Item, Rect, State)
  else
  begin
    Canvas.FillRect(Rect);
    WideCanvasTextOut(Canvas, Rect.Left + 2, Rect.Top, Item.Caption);
  end;
end;

{$IFNDEF VER130}
procedure TTntCustomListView.CopySelection(Destination: TCustomListControl);
var
  I: Integer;
begin
  for I := 0 to Items.Count - 1 do
    if Items[I].Selected then
      WideListControl_AddItem(Destination, Items[I].Caption, Items[I].Data);
end;
{$ENDIF}

procedure TTntCustomListView.AddItem(const Item: WideString; AObject: TObject);
begin
  with Items.Add do
  begin
    Caption := Item;
    Data := AObject;
  end;
end;

//-------------

function TTntCustomListView.FindCaption(StartIndex: Integer; Value: WideString;
  Partial, Inclusive, Wrap: Boolean): TTntListItem;
const
  FullString: array[Boolean] of Integer = (0, LVFI_PARTIAL);
  Wraps: array[Boolean] of Integer = (0, LVFI_WRAP);
var
  Info: TLVFindInfoW;
  Index: Integer;
begin
  if (not Win32PlatformIsUnicode) then
    Result := inherited FindCaption(StartIndex, Value, Partial, Inclusive, Wrap) as TTntListItem
  else begin
    with Info do
    begin
      flags := LVFI_STRING or FullString[Partial] or Wraps[Wrap];
      psz := PWideChar(Value);
    end;
    if Inclusive then Dec(StartIndex);
    Index := SendMessageW(Handle, LVM_FINDITEMW, StartIndex, Longint(@Info));
    if Index <> -1 then Result := Items[Index]
    else Result := nil;
  end;
end;

function TTntCustomListView.StringWidth(S: WideString): Integer;
begin
  if (not Win32PlatformIsUnicode) then
    Result := inherited StringWidth(S)
  else
    Result := SendMessageW(Handle, LVM_GETSTRINGWIDTHW, 0, Longint(PWideChar(S)))
end;

function TTntCustomListView.GetSearchString: WideString;
var
  Buffer: array[0..1023] of WideChar;
begin
  if (not Win32PlatformIsUnicode) then
    Result := inherited GetSearchString
  else begin
    Result := '';
    if HandleAllocated
    and Bool(SendMessageW(Handle, LVM_GETISEARCHSTRINGW, 0, Longint(PWideChar(@Buffer[0])))) then
      Result := Buffer;
  end;
end;

function TTntCustomListView.IsHintStored: Boolean;
begin
  Result := TntControl_IsHintStored(Self);
end;

function TTntCustomListView.GetHint: WideString;
begin
  Result := TntControl_GetHint(Self)
end;

procedure TTntCustomListView.SetHint(const Value: WideString);
begin
  TntControl_SetHint(Self, Value);
end;

procedure TTntCustomListView.ActionChange(Sender: TObject; CheckDefaults: Boolean);
begin
  TntControl_BeforeInherited_ActionChange(Self, Sender, CheckDefaults);
  inherited;
end;

function TTntCustomListView.GetActionLinkClass: TControlActionLinkClass;
begin
  Result := TntControl_GetActionLinkClass(Self, inherited GetActionLinkClass);
end;

{ TTntRichEditStrings }
type
  TTntRichEditStrings = class(TTntMemoStrings)
  private
    RichEdit: TCustomRichEdit{TNT-ALLOW TCustomRichEdit};
    procedure EnableChange(const Value: Boolean);
  protected
    procedure SetTextStr(const Value: WideString); override;
  public
    constructor Create;
    procedure AddStrings(Strings: TStrings{TNT-ALLOW TStrings}); overload; override;
    procedure AddStrings(Strings: TWideStrings); overload; override;
    //--
    procedure LoadFromStream(Stream: TStream); override;
{$IFDEF JCL}
    procedure SaveToStream(Stream: TStream; WithBOM: Boolean = True); override;
{$ELSE}
    procedure SaveToStream(Stream: TStream); override;
{$ENDIF}
    procedure LoadFromFile(const FileName: WideString); override;
    procedure SaveToFile(const FileName: WideString); override;
  end;

{ TTntRichEditStrings }

constructor TTntRichEditStrings.Create;
begin
  inherited Create;
  FRichEditMode := True;
end;

procedure TTntRichEditStrings.AddStrings(Strings: TStrings{TNT-ALLOW TStrings});
var
  SelChange: TNotifyEvent;
begin
  SelChange := TTntCustomRichEdit(RichEdit).OnSelectionChange;
  TTntCustomRichEdit(RichEdit).OnSelectionChange := nil;
  try
    inherited;
  finally
    TTntCustomRichEdit(RichEdit).OnSelectionChange := SelChange;
  end;
end;

procedure TTntRichEditStrings.AddStrings(Strings: TWideStrings);
var
  SelChange: TNotifyEvent;
begin
  SelChange := TTntCustomRichEdit(RichEdit).OnSelectionChange;
  TTntCustomRichEdit(RichEdit).OnSelectionChange := nil;
  try
    inherited;
  finally
    TTntCustomRichEdit(RichEdit).OnSelectionChange := SelChange;
  end;
end;

procedure TTntRichEditStrings.EnableChange(const Value: Boolean);
var
  EventMask: Longint;
begin
  with RichEdit do
  begin
    if Value then
      EventMask := SendMessage(Handle, EM_GETEVENTMASK, 0, 0) or ENM_CHANGE
    else
      EventMask := SendMessage(Handle, EM_GETEVENTMASK, 0, 0) and not ENM_CHANGE;
    SendMessage(Handle, EM_SETEVENTMASK, 0, EventMask);
  end;
end;

procedure TTntRichEditStrings.SetTextStr(const Value: WideString);
begin
  EnableChange(False);
  try
    inherited;
  finally
    EnableChange(True);
  end;
end;

type TAccessCustomRichEdit = class(TCustomRichEdit{TNT-ALLOW TCustomRichEdit});

procedure TTntRichEditStrings.LoadFromStream(Stream: TStream);
begin
  TAccessCustomRichEdit(RichEdit).Lines.LoadFromStream(Stream);
end;

{$IFDEF JCL}
procedure TTntRichEditStrings.SaveToStream(Stream: TStream; WithBOM: Boolean = True);
{$ELSE}
procedure TTntRichEditStrings.SaveToStream(Stream: TStream);
{$ENDIF}
begin
  TAccessCustomRichEdit(RichEdit).Lines.SaveToStream(Stream);
end;

procedure TTntRichEditStrings.LoadFromFile(const FileName: WideString);
begin
  TAccessCustomRichEdit(RichEdit).Lines.LoadFromFile(FileName);
end;

procedure TTntRichEditStrings.SaveToFile(const FileName: WideString);
begin
  TAccessCustomRichEdit(RichEdit).Lines.SaveToFile(FileName);
end;

{ TTntCustomRichEdit }

constructor TTntCustomRichEdit.Create(AOwner: TComponent);
begin
  inherited;
  FRichEditStrings := TTntRichEditStrings.Create;
  TTntRichEditStrings(FRichEditStrings).Memo := Self;
  TTntRichEditStrings(FRichEditStrings).RichEdit := Self;
  TTntRichEditStrings(FRichEditStrings).LineBreakStyle := Self.LineBreakStyle;
end;

var
  FRichEdit20Module: THandle = 0;

function IsRichEdit20Available: Boolean;
const
  RICHED20_DLL = 'RICHED20.DLL';
begin
  if FRichEdit20Module = 0 then
    FRichEdit20Module := LoadLibrary(RICHED20_DLL);
  Result := FRichEdit20Module <> 0;
end;

procedure TTntCustomRichEdit.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  if WordWrap then
    Params.Style := Params.Style and not WS_HSCROLL; // more compatible with RichEdit 1.0
end;

procedure TTntCustomRichEdit.CreateWindowHandle(const Params: TCreateParams);
begin
  if Win32PlatformIsUnicode and IsRichEdit20Available then
    CreateUnicodeHandle(Self, Params, RICHEDIT_CLASSW)
  else
    inherited
end;

var
  AIMM: IActiveIMMApp = nil;

function EnableActiveIMM: Boolean;
begin
  if AIMM <> nil then
    Result := True
  else begin
    Result := False;
    try
      if ClassIsRegistered(CLASS_CActiveIMM) then begin
        AIMM := CoCActiveIMM.Create;
        AIMM.Activate(1);
        Result := True;
      end;
    except
      AIMM := nil;
    end;
  end;
end;

procedure TTntCustomRichEdit.CreateWnd;
const
  EM_SETEDITSTYLE = WM_USER + 204;
  SES_USEAIMM = 64;
begin
  inherited;
  // Only supported in RichEdit 3.0, but this flag is harmless to RichEdit1.0 or RichEdit 2.0
  if EnableActiveIMM then
    SendMessage(Handle, EM_SETEDITSTYLE, SES_USEAIMM, SES_USEAIMM);
end;

procedure TTntCustomRichEdit.DefineProperties(Filer: TFiler);
begin
  inherited;
  DefineWideProperties(Filer, Self);
end;

destructor TTntCustomRichEdit.Destroy;
begin
  FreeAndNil(FRichEditStrings);
  inherited;
end;

function TTntCustomRichEdit.LineBreakStyle: TTntTextLineBreakStyle;
begin
  if Win32PlatformIsUnicode and IsRichEdit20Available then
    Result := tlbsCR
  else
    Result := tlbsCRLF;
end;

procedure TTntCustomRichEdit.SetRichEditStrings(const Value: TTntWideStrings);
begin
  FRichEditStrings.Assign(Value);
end;

function TTntCustomRichEdit.GetSelText: string{TNT-ALLOW string};
begin
  if (not IsWindowUnicode(Handle)) then begin
    Result := TntAdjustLineBreaks(inherited SelText, tlbsCRLF)
  end else
    Result := GetWideSelText;
end;

function TTntCustomRichEdit.GetWideSelText: WideString;
var
  CharRange: TCharRange;
  Length: Integer;
begin
  if (not IsWindowUnicode(Handle)) then
    Result := inherited SelText
  else begin
    SendMessageW(Handle, EM_EXGETSEL, 0, Longint(@CharRange));
    SetLength(Result, CharRange.cpMax - CharRange.cpMin + 1);
    Length := SendMessageW(Handle, EM_GETSELTEXT, 0, Longint(PWideChar(Result)));
    SetLength(Result, Length);
  end;
  if LineBreakStyle <> tlbsCRLF then
    Result := TntAdjustLineBreaks(Result, tlbsCRLF)
end;

procedure TTntCustomRichEdit.SetWideSelText(const Value: WideString);
begin
  TntCustomEdit_SetSelText(Self, TntAdjustLineBreaks(Value, LineBreakStyle));
end;

function TTntCustomRichEdit.GetText: WideString;
begin
  Result := TntControl_GetText(Self);
  if IsWindowUnicode(WindowHandle) and (LineBreakStyle <> tlbsCRLF) then
    Result := TntAdjustLineBreaks(Result, tlbsCRLF);
end;

procedure TTntCustomRichEdit.SetText(const Value: WideString);
begin
  if (not IsWindowUnicode(WindowHandle)) then
    TntControl_SetText(Self, Value)
  else
    TntControl_SetText(Self, TntAdjustLineBreaks(Value, LineBreakStyle));
end;

function TTntCustomRichEdit.IsHintStored: Boolean;
begin
  Result := TntControl_IsHintStored(Self);
end;

function TTntCustomRichEdit.GetHint: WideString;
begin
  Result := TntControl_GetHint(Self);
end;

procedure TTntCustomRichEdit.SetHint(const Value: WideString);
begin
  TntControl_SetHint(Self, Value);
end;

procedure TTntCustomRichEdit.WMGetTextLength(var Message: TWMGetTextLength);
begin
  if FPrintingTextLength <> 0 then
    Message.Result := FPrintingTextLength
  else
    inherited;
end;

procedure TTntCustomRichEdit.Print(const Caption: string{TNT-ALLOW string});
begin
  if (LineBreakStyle <> tlbsCRLF) then
    FPrintingTextLength := TntAdjustLineBreaksLength(Text, LineBreakStyle)
  else
    FPrintingTextLength := 0;
  try
    inherited
  finally
    FPrintingTextLength := 0;
  end;
end;

function TTntCustomRichEdit.CharPosToGet(CharPos: Integer): Integer;
var
  i: Integer;
  ThisLine: Integer;
  CharCount: Integer;
  LineStart: Integer;
  NumLineBreaks: Integer;
begin
  if (LineBreakStyle = tlbsCRLF) or (CharPos <= 0) then
    Result := CharPos
  else begin
    Assert(Win32PlatformIsUnicode);
    ThisLine := SendMessageW(Handle, EM_EXLINEFROMCHAR, 0, CharPos);
    CharCount := 0;
    for i := 0 to ThisLine - 1 do
      Inc(CharCount, TntMemo_LineLength(Handle, i));
    LineStart := TntMemo_LineStart(Handle, ThisLine);
    NumLineBreaks := LineStart - CharCount;
    Result := CharPos + NumLineBreaks; {inflate CR -> CR/LF}
  end;
end;

function TTntCustomRichEdit.CharPosToSet(CharPos: Integer): Integer;
var
  Line: Integer;
  NumLineBreaks: Integer;
  CharCount: Integer;
  LineStart: Integer;
  LineLength: Integer;
begin
  if (LineBreakStyle = tlbsCRLF) or (CharPos <= 0) then
    Result := CharPos
  else begin
    Assert(Win32PlatformIsUnicode);
    NumLineBreaks := 0;
    CharCount := 0;
    for Line := 0 to Lines.Count - 1 do begin
      LineStart := TntMemo_LineStart(Handle, Line);
      if CharPos < (LineStart + NumLineBreaks) then
        break; {found it (it must have been the line separator)}
      if LineStart > CharCount then begin
        Inc(NumLineBreaks);
        Inc(CharCount);
      end;
      LineLength := TntMemo_LineLength(Handle, Line);
      Inc(CharCount, LineLength);
      if (CharPos >= (LineStart + NumLineBreaks))
      and (CharPos < (LineStart + LineLength + NumLineBreaks)) then
        break; {found it}
    end;
    Result := CharPos - NumLineBreaks; {deflate CR/LF -> CR}
  end;
end;

function TTntCustomRichEdit.GetSelLength: Integer;
var
  CharRange: TCharRange;
begin
  if (not IsWindowUnicode(Handle))
  or (LineBreakStyle = tlbsCRLF) then
    Result := inherited GetSelLength
  else begin
    SendMessageW(Handle, EM_EXGETSEL, 0, Longint(@CharRange));
    Result := CharPosToGet(CharRange.cpMax) - CharPosToGet(CharRange.cpMin);
  end;
end;

function TTntCustomRichEdit.GetSelStart: Integer;
begin
  Result := inherited GetSelStart;
  if IsWindowUnicode(Handle)
  and (LineBreakStyle <> tlbsCRLF) then
    Result := CharPosToGet(Result);
end;

procedure TTntCustomRichEdit.SetSelStart(Value: Integer);
begin
  if (not IsWindowUnicode(Handle))
  or (LineBreakStyle = tlbsCRLF) then
    inherited SetSelStart(Value)
  else
    inherited SetSelStart(CharPosToSet(Value));
end;

procedure TTntCustomRichEdit.SetSelLength(Value: Integer);
var
  SelStart: Integer;
  SelEnd: Integer;
begin
  if (not IsWindowUnicode(Handle))
  or (LineBreakStyle = tlbsCRLF) then
    inherited SetSelLength(Value)
  else begin
    SelStart := Self.SelStart;
    SelEnd := SelStart + Value;
    inherited SetSelLength(CharPosToSet(SelEnd) - CharPosToSet(SelStart));
  end;
end;

procedure TTntCustomRichEdit.ActionChange(Sender: TObject; CheckDefaults: Boolean);
begin
  TntControl_BeforeInherited_ActionChange(Self, Sender, CheckDefaults);
  inherited;
end;

function TTntCustomRichEdit.GetActionLinkClass: TControlActionLinkClass;
begin
  Result := TntControl_GetActionLinkClass(Self, inherited GetActionLinkClass);
end;

{ TTntTabStrings }

type TAccessCustomTabControl = class(TCustomTabControl{TNT-ALLOW TCustomTabControl});

type
  TTntTabStrings = class(TTntWideStrings)
  private
    FTabControl: TCustomTabControl{TNT-ALLOW TCustomTabControl};
    FAnsiTabs: TStrings{TNT-ALLOW TStrings};
  protected
    function Get(Index: Integer): WideString; override;
    function GetCount: Integer; override;
    function GetObject(Index: Integer): TObject; override;
    procedure Put(Index: Integer; const S: WideString); override;
    procedure PutObject(Index: Integer; AObject: TObject); override;
    procedure SetUpdateState(Updating: Boolean); override;
  public
    procedure Clear; override;
    procedure Delete(Index: Integer); override;
    procedure Insert(Index: Integer; const S: WideString); override;
  end;

procedure TabControlError(const S: WideString);
begin
  raise EListError.Create(S);
end;

procedure TTntTabStrings.Clear;
begin
  FAnsiTabs.Clear;
end;

procedure TTntTabStrings.Delete(Index: Integer);
begin
  FAnsiTabs.Delete(Index);
end;

function TTntTabStrings.GetCount: Integer;
begin
  Result := FAnsiTabs.Count;
end;

function TTntTabStrings.GetObject(Index: Integer): TObject;
begin
  Result := FAnsiTabs.Objects[Index];
end;

procedure TTntTabStrings.PutObject(Index: Integer; AObject: TObject);
begin
  FAnsiTabs.Objects[Index] := AObject;
end;

procedure TTntTabStrings.SetUpdateState(Updating: Boolean);
begin
  inherited;
  TAccessStrings(FAnsiTabs).SetUpdateState(Updating);
end;

function TTntTabStrings.Get(Index: Integer): WideString;
const
  RTL: array[Boolean] of LongInt = (0, TCIF_RTLREADING);
var
  TCItem: TTCItemW;
  Buffer: array[0..4095] of WideChar;
begin
  if (not Win32PlatformIsUnicode) then
    Result := FAnsiTabs[Index]
  else begin
    TCItem.mask := TCIF_TEXT or RTL[FTabControl.UseRightToLeftReading];
    TCItem.pszText := Buffer;
    TCItem.cchTextMax := SizeOf(Buffer);
    if SendMessageW(FTabControl.Handle, TCM_GETITEMW, Index, Longint(@TCItem)) = 0 then
      TabControlError(WideFormat(sTabFailRetrieve, [Index]));
    Result := Buffer;
  end;
end;

function GetTabControlImageIndex(Self: TCustomTabControl{TNT-ALLOW TCustomTabControl}; TabIndex: Integer): Integer;
begin
  Result := TabIndex;
  with TAccessCustomTabControl(Self) do
    if Assigned(OnGetImageIndex) then OnGetImageIndex(Self, TabIndex, Result);
end;

procedure TTntTabStrings.Put(Index: Integer; const S: WideString);
const
  RTL: array[Boolean] of LongInt = (0, TCIF_RTLREADING);
var
  TCItem: TTCItemW;
begin
  if (not Win32PlatformIsUnicode) then
    FAnsiTabs[Index] := S
  else begin
    TCItem.mask := TCIF_TEXT or RTL[FTabControl.UseRightToLeftReading] or TCIF_IMAGE;
    TCItem.pszText := PWideChar(S);
    TCItem.iImage := GetTabControlImageIndex(FTabControl, Index);
    if SendMessageW(FTabControl.Handle, TCM_SETITEMW, Index, Longint(@TCItem)) = 0 then
      TabControlError(WideFormat(sTabFailSet, [S, Index]));
    TAccessCustomTabControl(FTabControl).UpdateTabImages;
  end;
end;

procedure TTntTabStrings.Insert(Index: Integer; const S: WideString);
const
  RTL: array[Boolean] of LongInt = (0, TCIF_RTLREADING);
var
  TCItem: TTCItemW;
begin
  if (not Win32PlatformIsUnicode) then
    FAnsiTabs.Insert(Index, S)
  else begin
    TCItem.mask := TCIF_TEXT or RTL[FTabControl.UseRightToLeftReading] or TCIF_IMAGE;
    TCItem.pszText := PWideChar(S);
    TCItem.iImage := GetTabControlImageIndex(FTabControl, Index);
    if SendMessageW(FTabControl.Handle, TCM_INSERTITEMW, Index, Longint(@TCItem)) < 0 then
      TabControlError(WideFormat(sTabFailSet, [S, Index]));
    TAccessCustomTabControl(FTabControl).UpdateTabImages;
  end;
end;

{ TTntCustomTabControl }

constructor TTntCustomTabControl.Create(AOwner: TComponent);
begin
  inherited;
  FTabs := TTntTabStrings.Create;
  TTntTabStrings(FTabs).FTabControl := Self;
  TTntTabStrings(FTabs).FAnsiTabs := inherited Tabs;
end;

destructor TTntCustomTabControl.Destroy;
begin
  TTntTabStrings(FTabs).FTabControl := nil;
  TTntTabStrings(FTabs).FAnsiTabs := nil;
  FreeAndNil(FTabs);
  FreeAndNil(FSaveTabs);
  inherited;
end;

procedure TTntCustomTabControl.CreateWindowHandle(const Params: TCreateParams);
begin
  CreateUnicodeHandle_ComCtl(Self, Params, WC_TABCONTROL);
end;

procedure TTntCustomTabControl.DefineProperties(Filer: TFiler);
begin
  inherited;
  DefineWideProperties(Filer, Self);
end;

procedure TTntCustomTabControl.CreateWnd;
begin
  inherited;
  if FSaveTabs <> nil then
  begin
    FTabs.Assign(FSaveTabs);
    FreeAndNil(FSaveTabs);
    TabIndex := FSaveTabIndex;
  end;
end;

procedure TTntCustomTabControl.DestroyWnd;
begin
  if (FTabs <> nil) and (FTabs.Count > 0) then
  begin
    FSaveTabs := TTntWideStringList.Create;
    FSaveTabs.Assign(FTabs);
    FSaveTabIndex := TabIndex;
  end;
  inherited;
end;

procedure TTntCustomTabControl.SetTabs(const Value: TTntWideStrings);
begin
  FTabs.Assign(Value);
end;

function TTntCustomTabControl.IsHintStored: Boolean;
begin
  Result := TntControl_IsHintStored(Self);
end;

function TTntCustomTabControl.GetHint: WideString;
begin
  Result := TntControl_GetHint(Self);
end;

procedure TTntCustomTabControl.SetHint(const Value: WideString);
begin
  TntControl_SetHint(Self, Value);
end;

procedure TTntCustomTabControl.CMDialogChar(var Message: TCMDialogChar);
var
  I: Integer;
begin
  for I := 0 to Tabs.Count - 1 do
    if IsWideCharAccel(Message.CharCode, Tabs[I]) and CanShowTab(I) and CanFocus then
    begin
      Message.Result := 1;
      if CanChange then
      begin
        TabIndex := I;
        Change;
      end;
      Exit;
    end;
  Broadcast(Message);
end;

procedure TTntCustomTabControl.ActionChange(Sender: TObject; CheckDefaults: Boolean);
begin
  TntControl_BeforeInherited_ActionChange(Self, Sender, CheckDefaults);
  inherited;
end;

function TTntCustomTabControl.GetActionLinkClass: TControlActionLinkClass;
begin
  Result := TntControl_GetActionLinkClass(Self, inherited GetActionLinkClass);
end;

{ TTntTabSheet }

procedure TTntTabSheet.CreateWindowHandle(const Params: TCreateParams);
begin
  CreateUnicodeHandle(Self, Params, '');
end;

function TTntTabSheet.IsCaptionStored: Boolean;
begin
  Result := TntControl_IsCaptionStored(Self);
end;

function TTntTabSheet.GetCaption: TWideCaption;
begin
  Result := TntControl_GetText(Self);
end;

procedure TTntTabSheet.SetCaption(const Value: TWideCaption);
begin
  TntControl_SetText(Self, Value);
end;

procedure TTntTabSheet.DefineProperties(Filer: TFiler);
begin
  inherited;
  DefineWideProperties(Filer, Self);
end;

function TTntTabSheet.IsHintStored: Boolean;
begin
  Result := TntControl_IsHintStored(Self);
end;

function TTntTabSheet.GetHint: WideString;
begin
  Result := TntControl_GetHint(Self);
end;

procedure TTntTabSheet.SetHint(const Value: WideString);
begin
  TntControl_SetHint(Self, Value);
end;

procedure TTntTabSheet.WMSetText(var Message: TWMSetText);
begin
  if (not Win32PlatformIsUnicode)
  or (HandleAllocated)
  or (Message.Text = AnsiString(TntControl_GetText(Self)))
  or (Force_Inherited_WMSETTEXT) then
    inherited
  else begin
    // NT, handle not allocated and text is different
    Force_Inherited_WMSETTEXT := True;
    try
      TntControl_SetText(Self, Message.Text) { sync WideCaption with ANSI Caption }
    finally
      Force_Inherited_WMSETTEXT := FALSE;
    end;
  end;
end;

procedure TTntTabSheet.ActionChange(Sender: TObject; CheckDefaults: Boolean);
begin
  TntControl_BeforeInherited_ActionChange(Self, Sender, CheckDefaults);
  inherited;
end;

function TTntTabSheet.GetActionLinkClass: TControlActionLinkClass;
begin
  Result := TntControl_GetActionLinkClass(Self, inherited GetActionLinkClass);
end;

{ TTntPageControl }

procedure TTntPageControl.CreateWindowHandle(const Params: TCreateParams);
begin
  CreateUnicodeHandle_ComCtl(Self, Params, WC_TABCONTROL);
end;

procedure TTntPageControl.DefineProperties(Filer: TFiler);
begin
  inherited;
  DefineWideProperties(Filer, Self);
end;

function TTntPageControl.IsHintStored: Boolean;
begin
  Result := TntControl_IsHintStored(Self);
end;

function TTntPageControl.GetHint: WideString;
begin
  Result := TntControl_GetHint(Self);
end;

procedure TTntPageControl.SetHint(const Value: WideString);
begin
  TntControl_SetHint(Self, Value);
end;

procedure TTntPageControl.WndProc(var Message: TMessage);
const
  RTL: array[Boolean] of Cardinal = (0, TCIF_RTLREADING);
var
  TCItemA: PTCItemA;
  TabSheet: TTabSheet{TNT-ALLOW TTabSheet};
  Text: WideString;
begin
  if (not Win32PlatformIsUnicode) then
    inherited
  else begin
    case Message.Msg of
      TCM_SETITEMA:
        begin
          TCItemA := PTCItemA(Message.lParam);
          if ((TCItemA.mask and TCIF_PARAM) = TCIF_PARAM) then
            TabSheet := TObject(TCItemA.lParam) as TTabSheet{TNT-ALLOW TTabSheet}
          else if ((TCItemA.mask and TCIF_TEXT) = TCIF_TEXT)
          and (Message.wParam >= 0) and (Message.wParam <= Tabs.Count - 1) then
            TabSheet := Tabs.Objects[Message.wParam] as TTabSheet{TNT-ALLOW TTabSheet}
          else
            TabSheet := nil;

          if TabSheet = nil then begin
            // will always be followed by TCM_SETITEMA(TCIF_PARAM) if TCIF_TEXT is present
            TCItemA.mask := TCItemA.mask and (not TCIF_TEXT);
          end else begin
            // convert message to unicode, add text
            Message.Msg := TCM_SETITEMW;
            TCItemA.mask := TCItemA.mask or TCIF_TEXT or RTL[UseRightToLeftReading];
            if TabSheet is TTntTabSheet then
              Text := TTntTabSheet(TabSheet).Caption
            else
              Text := TabSheet.Caption;
            TCItemA.pszText := PAnsiChar(PWideChar(Text));
          end;
        end;
      TCM_INSERTITEMA:
        begin
          TCItemA := PTCItemA(Message.lParam);
          // will always be followed by TCM_SETITEMA(TCIF_PARAM) if TCIF_TEXT is present
          TCItemA.mask := TCItemA.mask and (not TCIF_TEXT);
        end;
    end;
    inherited;
  end;
end;

procedure TTntPageControl.CMDialogChar(var Message: TCMDialogChar);
var
  I: Integer;
  TabText: WideString;
begin
  for I := 0 to PageCount - 1 do begin
    if Pages[i] is TTntTabSheet then
      TabText := TTntTabSheet(Pages[i]).Caption
    else
      TabText := Pages[i].Caption;
    if IsWideCharAccel(Message.CharCode, TabText) and CanShowTab(Pages[i].TabIndex) and CanFocus then
    begin
      Message.Result := 1;
      if CanChange then
      begin
        TabIndex := Pages[i].TabIndex;
        Change;
      end;
      Exit;
    end;
  end;
  Broadcast(Message);
end;

procedure TTntPageControl.CMDockClient(var Message: TCMDockClient);
var
  IsVisible: Boolean;
  DockCtl: TControl;
begin
  Message.Result := 0;
  FNewDockSheet := TTntTabSheet.Create(Self);
  try
    try
      DockCtl := Message.DockSource.Control;
      if DockCtl is TTntForm{TNT-ALLOW TTntForm} then
        FNewDockSheet.Caption := TTntForm{TNT-ALLOW TTntForm}(DockCtl).Caption
      else if DockCtl is TCustomForm then
        FNewDockSheet.Caption := TCustomForm(DockCtl).Caption;
      FNewDockSheet.PageControl := Self;
      DockCtl.Dock(Self, Message.DockSource.DockRect);
    except
      FNewDockSheet.Free;
      raise;
    end;
    IsVisible := DockCtl.Visible;
    FNewDockSheet.TabVisible := IsVisible;
    if IsVisible then ActivePage := FNewDockSheet;
    DockCtl.Align := alClient;
  finally
    FNewDockSheet := nil;
  end;
end;

procedure TTntPageControl.DoAddDockClient(Client: TControl; const ARect: TRect);
begin
  if FNewDockSheet <> nil then
    Client.Parent := FNewDockSheet;
end;

procedure TTntPageControl.CMDockNotification(var Message: TCMDockNotification);
var
  I: Integer;
  S: WideString;
  Page: TTabSheet{TNT-ALLOW TTabSheet};
begin
  Page := GetPageFromDockClient(Message.Client);
  if (Message.NotifyRec.ClientMsg <> WM_SETTEXT)
  or (Page = nil) or (not (Page is TTntTabSheet)) then
    inherited
  else begin
    if (Message.Client is TWinControl)
    and (TWinControl(Message.Client).HandleAllocated)
    and IsWindowUnicode(TWinControl(Message.Client).Handle) then
      S := PWideChar(Message.NotifyRec.MsgLParam)
    else
      S := PAnsiChar(Message.NotifyRec.MsgLParam);
    { Search for first CR/LF and end string there }
    for I := 1 to Length(S) do
      if S[I] in [CR, LF] then
      begin
        SetLength(S, I - 1);
        Break;
      end;
    TTntTabSheet(Page).Caption := S;
  end;
end;

procedure TTntPageControl.ActionChange(Sender: TObject; CheckDefaults: Boolean);
begin
  TntControl_BeforeInherited_ActionChange(Self, Sender, CheckDefaults);
  inherited;
end;

function TTntPageControl.GetActionLinkClass: TControlActionLinkClass;
begin
  Result := TntControl_GetActionLinkClass(Self, inherited GetActionLinkClass);
end;

{ TTntTrackBar }

procedure TTntTrackBar.CreateWindowHandle(const Params: TCreateParams);
begin
  CreateUnicodeHandle_ComCtl(Self, Params, TRACKBAR_CLASS);
end;

procedure TTntTrackBar.DefineProperties(Filer: TFiler);
begin
  inherited;
  DefineWideProperties(Filer, Self);
end;

function TTntTrackBar.IsHintStored: Boolean;
begin
  Result := TntControl_IsHintStored(Self);
end;

function TTntTrackBar.GetHint: WideString;
begin
  Result := TntControl_GetHint(Self);
end;

procedure TTntTrackBar.SetHint(const Value: WideString);
begin
  TntControl_SetHint(Self, Value);
end;

procedure TTntTrackBar.ActionChange(Sender: TObject; CheckDefaults: Boolean);
begin
  TntControl_BeforeInherited_ActionChange(Self, Sender, CheckDefaults);
  inherited;
end;

function TTntTrackBar.GetActionLinkClass: TControlActionLinkClass;
begin
  Result := TntControl_GetActionLinkClass(Self, inherited GetActionLinkClass);
end;

{ TTntProgressBar }

procedure TTntProgressBar.CreateWindowHandle(const Params: TCreateParams);
begin
  CreateUnicodeHandle_ComCtl(Self, Params, PROGRESS_CLASS);
end;

procedure TTntProgressBar.DefineProperties(Filer: TFiler);
begin
  inherited;
  DefineWideProperties(Filer, Self);
end;

function TTntProgressBar.IsHintStored: Boolean;
begin
  Result := TntControl_IsHintStored(Self);
end;

function TTntProgressBar.GetHint: WideString;
begin
  Result := TntControl_GetHint(Self);
end;

procedure TTntProgressBar.SetHint(const Value: WideString);
begin
  TntControl_SetHint(Self, Value);
end;

procedure TTntProgressBar.ActionChange(Sender: TObject; CheckDefaults: Boolean);
begin
  TntControl_BeforeInherited_ActionChange(Self, Sender, CheckDefaults);
  inherited;
end;

function TTntProgressBar.GetActionLinkClass: TControlActionLinkClass;
begin
  Result := TntControl_GetActionLinkClass(Self, inherited GetActionLinkClass);
end;

{ TTntCustomUpDown }

procedure TTntCustomUpDown.CreateWindowHandle(const Params: TCreateParams);
begin
  CreateUnicodeHandle_ComCtl(Self, Params, UPDOWN_CLASS);
end;

procedure TTntCustomUpDown.DefineProperties(Filer: TFiler);
begin
  inherited;
  DefineWideProperties(Filer, Self);
end;

function TTntCustomUpDown.IsHintStored: Boolean;
begin
  Result := TntControl_IsHintStored(Self);
end;

function TTntCustomUpDown.GetHint: WideString;
begin
  Result := TntControl_GetHint(Self);
end;

procedure TTntCustomUpDown.SetHint(const Value: WideString);
begin
  TntControl_SetHint(Self, Value);
end;

procedure TTntCustomUpDown.ActionChange(Sender: TObject; CheckDefaults: Boolean);
begin
  TntControl_BeforeInherited_ActionChange(Self, Sender, CheckDefaults);
  inherited;
end;

function TTntCustomUpDown.GetActionLinkClass: TControlActionLinkClass;
begin
  Result := TntControl_GetActionLinkClass(Self, inherited GetActionLinkClass);
end;

{ TTntDateTimePicker }

procedure TTntDateTimePicker.CreateWindowHandle(const Params: TCreateParams);
begin
  CreateUnicodeHandle_ComCtl(Self, Params, DATETIMEPICK_CLASS);
end;

procedure TTntDateTimePicker.DefineProperties(Filer: TFiler);
begin
  inherited;
  DefineWideProperties(Filer, Self);
end;

function TTntDateTimePicker.IsHintStored: Boolean;
begin
  Result := TntControl_IsHintStored(Self);
end;

function TTntDateTimePicker.GetHint: WideString;
begin
  Result := TntControl_GetHint(Self);
end;

procedure TTntDateTimePicker.SetHint(const Value: WideString);
begin
  TntControl_SetHint(Self, Value);
end;

procedure TTntDateTimePicker.ActionChange(Sender: TObject; CheckDefaults: Boolean);
begin
  TntControl_BeforeInherited_ActionChange(Self, Sender, CheckDefaults);
  inherited;
end;

function TTntDateTimePicker.GetActionLinkClass: TControlActionLinkClass;
begin
  Result := TntControl_GetActionLinkClass(Self, inherited GetActionLinkClass);
end;

{ TTntMonthCalendar }

procedure TTntMonthCalendar.CreateWindowHandle(const Params: TCreateParams);
begin
  CreateUnicodeHandle_ComCtl(Self, Params, MONTHCAL_CLASS);
  if Win32PlatformIsUnicode then begin
    { For some reason WM_NOTIFY:MCN_GETDAYSTATE never gets called. }
    ForceGetMonthInfo;
  end;
end;

procedure TTntMonthCalendar.ForceGetMonthInfo;
var
  Hdr: TNMDayState;
  Days: array of TMonthDayState;
  Range: array[1..2] of TSystemTime;
begin
  // populate Days array
  Hdr.nmhdr.hwndFrom := Handle;
  Hdr.nmhdr.idFrom := 0;
  Hdr.nmhdr.code := MCN_GETDAYSTATE;
  Hdr.cDayState := MonthCal_GetMonthRange(Handle, GMR_DAYSTATE, @Range[1]);
  Hdr.stStart := Range[1];
  SetLength(Days, Hdr.cDayState);
  Hdr.prgDayState := @Days[0];
  SendMessage(Handle, CN_NOTIFY, Handle, Integer(@Hdr));
  // update day state
  SendMessage(Handle, MCM_SETDAYSTATE, Hdr.cDayState, Longint(Hdr.prgDayState))
end;

procedure TTntMonthCalendar.DefineProperties(Filer: TFiler);
begin
  inherited;
  DefineWideProperties(Filer, Self);
end;

function TTntMonthCalendar.IsHintStored: Boolean;
begin
  Result := TntControl_IsHintStored(Self);
end;

function TTntMonthCalendar.GetHint: WideString;
begin
  Result := TntControl_GetHint(Self);
end;

procedure TTntMonthCalendar.SetHint(const Value: WideString);
begin
  TntControl_SetHint(Self, Value);
end;

function TTntMonthCalendar.GetDate: TDate;
begin
  Result := Trunc(inherited Date); { Fixes issue where Date always reflects time of saving dfm. }
end;

procedure TTntMonthCalendar.SetDate(const Value: TDate);
begin
  inherited Date := Trunc(Value);
end;

procedure TTntMonthCalendar.ActionChange(Sender: TObject; CheckDefaults: Boolean);
begin
  TntControl_BeforeInherited_ActionChange(Self, Sender, CheckDefaults);
  inherited;
end;

function TTntMonthCalendar.GetActionLinkClass: TControlActionLinkClass;
begin
  Result := TntControl_GetActionLinkClass(Self, inherited GetActionLinkClass);
end;

{ TTntPageScroller }

procedure TTntPageScroller.CreateWindowHandle(const Params: TCreateParams);
begin
  CreateUnicodeHandle_ComCtl(Self, Params, WC_PAGESCROLLER);
end;

procedure TTntPageScroller.DefineProperties(Filer: TFiler);
begin
  inherited;
  DefineWideProperties(Filer, Self);
end;

function TTntPageScroller.IsHintStored: Boolean;
begin
  Result := TntControl_IsHintStored(Self);
end;

function TTntPageScroller.GetHint: WideString;
begin
  Result := TntControl_GetHint(Self);
end;

procedure TTntPageScroller.SetHint(const Value: WideString);
begin
  TntControl_SetHint(Self, Value);
end;

procedure TTntPageScroller.ActionChange(Sender: TObject; CheckDefaults: Boolean);
begin
  TntControl_BeforeInherited_ActionChange(Self, Sender, CheckDefaults);
  inherited;
end;

function TTntPageScroller.GetActionLinkClass: TControlActionLinkClass;
begin
  Result := TntControl_GetActionLinkClass(Self, inherited GetActionLinkClass);
end;

initialization
  RegisterClasses([TTntListItems]);
  RegisterClasses([TTntListItem]);
  RegisterClasses([TTntTabSheet]);

finalization
  if Assigned(AIMM) then
    AIMM.Deactivate;
  if FRichEdit20Module <> 0 then
    FreeLibrary(FRichEdit20Module);

end.
