
{*******************************************************}
{    The Delphi Unicode Controls Project                }
{                                                       }
{      http://home.ccci.org/wolbrink                    }
{                                                       }
{ Copyright (c) 2002, Troy Wolbrink (wolbrink@ccci.org) }
{                                                       }
{*******************************************************}

unit TntGrids;

interface

uses Classes, TntClasses, Grids, Windows, Controls, Messages;

type
{TNT-WARN TInplaceEdit}
  TTntInplaceEdit = class(TInplaceEdit{TNT-ALLOW TInplaceEdit})
  private
    function GetText: WideString;
    procedure SetText(const Value: WideString);
  protected
    procedure UpdateContents; override;
    procedure CreateWindowHandle(const Params: TCreateParams); override;
  public
    property Text: WideString read GetText write SetText;
  end;

  TTntGetEditEvent = procedure (Sender: TObject; ACol, ARow: Longint; var Value: WideString) of object;
  TTntSetEditEvent = procedure (Sender: TObject; ACol, ARow: Longint; const Value: WideString) of object;

{TNT-WARN TCustomDrawGrid}
{$IFDEF VER130}
  _TTntInternalCustomDrawGrid = class(TDrawGrid{TNT-ALLOW TDrawGrid})
{$ELSE}
  _TTntInternalCustomDrawGrid = class(TCustomDrawGrid{TNT-ALLOW TCustomDrawGrid})
{$ENDIF}
  private
    FSettingEditText: Boolean;
    procedure InternalSetEditText(ACol, ARow: Longint; const Value: string{TNT-ALLOW string}); dynamic; abstract;
  protected
    procedure SetEditText(ACol, ARow: Longint; const Value: string{TNT-ALLOW string}); override;
  end;

  TTntCustomDrawGrid = class(_TTntInternalCustomDrawGrid)
  private
    FOnGetEditText: TTntGetEditEvent;
    FOnSetEditText: TTntSetEditEvent;
    function GetHint: WideString;
    procedure SetHint(const Value: WideString);
    function IsHintStored: Boolean;
    procedure WMChar(var Msg: TWMChar); message WM_CHAR;
  protected
    function CreateEditor: TInplaceEdit{TNT-ALLOW TInplaceEdit}; override;
    procedure InternalSetEditText(ACol, ARow: Longint; const Value: string{TNT-ALLOW string}); override;
    function GetEditText(ACol, ARow: Longint): WideString; reintroduce; virtual;
    procedure SetEditText(ACol, ARow: Longint; const Value: WideString); reintroduce; virtual;
  protected
    procedure CreateWindowHandle(const Params: TCreateParams); override;
    procedure ShowEditorChar(Ch: WideChar); dynamic;
    procedure DefineProperties(Filer: TFiler); override;
    function GetActionLinkClass: TControlActionLinkClass; override;
    procedure ActionChange(Sender: TObject; CheckDefaults: Boolean); override;
    property OnGetEditText: TTntGetEditEvent read FOnGetEditText write FOnGetEditText;
    property OnSetEditText: TTntSetEditEvent read FOnSetEditText write FOnSetEditText;
  published
    property Hint: WideString read GetHint write SetHint stored IsHintStored;
  end;

{TNT-WARN TDrawGrid}
  TTntDrawGrid = class(TTntCustomDrawGrid)
  published
    property Align;
    property Anchors;
    property BiDiMode;
    property BorderStyle;
    property Color;
    property ColCount;
    property Constraints;
    property Ctl3D;
    property DefaultColWidth;
    property DefaultRowHeight;
    property DefaultDrawing;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property FixedColor;
    property FixedCols;
    property RowCount;
    property FixedRows;
    property Font;
    property GridLineWidth;
    property Options;
    property ParentBiDiMode;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ScrollBars;
    property ShowHint;
    property TabOrder;
    property Visible;
    property VisibleColCount;
    property VisibleRowCount;
    property OnClick;
    property OnColumnMoved;
    property OnContextPopup;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnDrawCell;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnGetEditMask;
    property OnGetEditText;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnMouseWheelDown;
    property OnMouseWheelUp;
    property OnRowMoved;
    property OnSelectCell;
    property OnSetEditText;
    property OnStartDock;
    property OnStartDrag;
    property OnTopLeftChanged;
  end;

  TTntStringGrid = class;

{TNT-WARN TStringGridStrings}
  TTntStringGridStrings = class(TTntWideStrings)
  private
    FIsCol: Boolean;
    FColRowIndex: Integer;
    FGrid: TTntStringGrid;
    function GetAnsiStrings: TStrings{TNT-ALLOW TStrings};
  protected
    function Get(Index: Integer): WideString; override;
    procedure Put(Index: Integer; const S: WideString); override;
    function GetCount: Integer; override;
    function GetObject(Index: Integer): TObject; override;
    procedure PutObject(Index: Integer; AObject: TObject); override;
    procedure SetUpdateState(Updating: Boolean); override;
  public
    constructor Create(AGrid: TTntStringGrid; AIndex: Longint);
    function Add(const S: WideString): Integer; override;
    procedure Assign(Source: TPersistent); override;
    procedure Clear; override;
    procedure Delete(Index: Integer); override;
    procedure Insert(Index: Integer; const S: WideString); override;
  end;

{TNT-WARN TStringGrid}
  _TTntInternalStringGrid = class(TStringGrid{TNT-ALLOW TStringGrid})
  private
    FSettingEditText: Boolean;
    procedure InternalSetEditText(ACol, ARow: Longint; const Value: string{TNT-ALLOW string}); dynamic; abstract;
  protected
    procedure SetEditText(ACol, ARow: Longint; const Value: string{TNT-ALLOW string}); override;
  end;

  TTntStringGrid = class(_TTntInternalStringGrid)
  private
    FCreatedRowStrings: TStringList{TNT-ALLOW TStringList};
    FCreatedColStrings: TStringList{TNT-ALLOW TStringList};
    FOnGetEditText: TTntGetEditEvent;
    FOnSetEditText: TTntSetEditEvent;
    function GetHint: WideString;
    procedure SetHint(const Value: WideString);
    function IsHintStored: Boolean;
    procedure WMChar(var Msg: TWMChar); message WM_CHAR;
    function GetCells(ACol, ARow: Integer): WideString;
    procedure SetCells(ACol, ARow: Integer; const Value: WideString);
    function FindGridStrings(IsCol: Boolean; ListIndex: Integer): TTntWideStrings;
    function GetCols(Index: Integer): TTntWideStrings;
    function GetRows(Index: Integer): TTntWideStrings;
    procedure SetCols(Index: Integer; const Value: TTntWideStrings);
    procedure SetRows(Index: Integer; const Value: TTntWideStrings);
  protected
    function CreateEditor: TInplaceEdit{TNT-ALLOW TInplaceEdit}; override;
    procedure DrawCell(ACol, ARow: Longint; ARect: TRect; AState: TGridDrawState); override;
    procedure InternalSetEditText(ACol, ARow: Longint; const Value: string{TNT-ALLOW string}); override;
    function GetEditText(ACol, ARow: Longint): WideString; reintroduce; virtual;
    procedure SetEditText(ACol, ARow: Longint; const Value: WideString); reintroduce; virtual;
  protected
    procedure CreateWindowHandle(const Params: TCreateParams); override;
    procedure ShowEditorChar(Ch: WideChar); dynamic;
    procedure DefineProperties(Filer: TFiler); override;
    function GetActionLinkClass: TControlActionLinkClass; override;
    procedure ActionChange(Sender: TObject; CheckDefaults: Boolean); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property Cells[ACol, ARow: Integer]: WideString read GetCells write SetCells;
    property Cols[Index: Integer]: TTntWideStrings read GetCols write SetCols;
    property Rows[Index: Integer]: TTntWideStrings read GetRows write SetRows;
  published
    property Hint: WideString read GetHint write SetHint stored IsHintStored;
    property OnGetEditText: TTntGetEditEvent read FOnGetEditText write FOnGetEditText;
    property OnSetEditText: TTntSetEditEvent read FOnSetEditText write FOnSetEditText;
  end;

implementation

uses SysUtils, TntGraphics, TntControls, TntStdCtrls, TntWideStrPropHelper,
  {$IFDEF JCL} JclUnicode, {$ELSE} Unicode, {$ENDIF} TntActnList;

{ TTntInplaceEdit }

procedure TTntInplaceEdit.CreateWindowHandle(const Params: TCreateParams);
begin
  TntCustomEdit_CreateWindowHandle(Self, Params);
end;

function TTntInplaceEdit.GetText: WideString;
begin
  Result := TntControl_GetText(Self);
end;

procedure TTntInplaceEdit.SetText(const Value: WideString);
begin
  TntControl_SetText(Self, Value);
end;

type TAccessCustomGrid = class(TCustomGrid);

procedure TTntInplaceEdit.UpdateContents;
begin
  Text := '';
  with TAccessCustomGrid(Grid) do
    Self.EditMask := GetEditMask(Col, Row);
  if (Grid is TTntStringGrid) then
    with (Grid as TTntStringGrid) do
      Self.Text := GetEditText(Col, Row)
  else if (Grid is TTntCustomDrawGrid) then
    with (Grid as TTntCustomDrawGrid) do
      Self.Text := GetEditText(Col, Row)
  else
    with TAccessCustomGrid(Grid) do
      Self.Text := GetEditText(Col, Row);
  with TAccessCustomGrid(Grid) do
    Self.MaxLength := GetEditLimit;
end;

{ _TTntInternalCustomDrawGrid }

procedure _TTntInternalCustomDrawGrid.SetEditText(ACol, ARow: Integer; const Value: string{TNT-ALLOW string});
begin
  if FSettingEditText then
    inherited
  else
    InternalSetEditText(ACol, ARow, Value);
end;


{ TTntCustomDrawGrid }

function TTntCustomDrawGrid.CreateEditor: TInplaceEdit{TNT-ALLOW TInplaceEdit};
begin
  Result := TTntInplaceEdit.Create(Self);
end;

procedure TTntCustomDrawGrid.CreateWindowHandle(const Params: TCreateParams);
begin
  CreateUnicodeHandle(Self, Params, '');
end;

procedure TTntCustomDrawGrid.DefineProperties(Filer: TFiler);
begin
  inherited;
  DefineWideProperties(Filer, Self);
end;

function TTntCustomDrawGrid.IsHintStored: Boolean;
begin
  Result := TntControl_IsHintStored(Self);
end;

function TTntCustomDrawGrid.GetHint: WideString;
begin
  Result := TntControl_GetHint(Self);
end;

procedure TTntCustomDrawGrid.SetHint(const Value: WideString);
begin
  TntControl_SetHint(Self, Value);
end;

function TTntCustomDrawGrid.GetEditText(ACol, ARow: Integer): WideString;
begin
  Result := '';
  if Assigned(FOnGetEditText) then FOnGetEditText(Self, ACol, ARow, Result);
end;

procedure TTntCustomDrawGrid.InternalSetEditText(ACol, ARow: Integer; const Value: string{TNT-ALLOW string});
begin
  if not FSettingEditText then
    SetEditText(ACol, ARow, TntControl_GetText(InplaceEditor));
end;

procedure TTntCustomDrawGrid.SetEditText(ACol, ARow: Integer; const Value: WideString);
begin
  if Assigned(FOnSetEditText) then FOnSetEditText(Self, ACol, ARow, Value);
end;

procedure TTntCustomDrawGrid.WMChar(var Msg: TWMChar);
begin
  if (goEditing in Options)
  and (AnsiChar(Msg.CharCode) in [^H, #32..#255]) then begin
    RestoreWMCharMsg(TMessage(Msg));
    ShowEditorChar(WideChar(Msg.CharCode));
  end else
    inherited;
end;

procedure TTntCustomDrawGrid.ShowEditorChar(Ch: WideChar);
begin
  ShowEditor;
  if InplaceEditor <> nil then begin
    if Win32PlatformIsUnicode then
      PostMessageW(InplaceEditor.Handle, WM_CHAR, Word(Ch), 0)
    else
      PostMessageA(InplaceEditor.Handle, WM_CHAR, Word(Ch), 0);
  end;
end;

procedure TTntCustomDrawGrid.ActionChange(Sender: TObject; CheckDefaults: Boolean);
begin
  TntControl_BeforeInherited_ActionChange(Self, Sender, CheckDefaults);
  inherited;
end;

function TTntCustomDrawGrid.GetActionLinkClass: TControlActionLinkClass;
begin
  Result := TntControl_GetActionLinkClass(Self, inherited GetActionLinkClass);
end;

{ TTntStringGridStrings }

procedure TTntStringGridStrings.Assign(Source: TPersistent);
var
  UTF8Strings: TStringList{TNT-ALLOW TStringList};
  i: integer;
begin
  UTF8Strings := TStringList{TNT-ALLOW TStringList}.Create;
  try
    if Source is TStrings{TNT-ALLOW TStrings} then begin
      for i := 0 to TStrings{TNT-ALLOW TStrings}(Source).Count - 1 do
        UTF8Strings.AddObject(WideStringToUTF8(WideString(TStrings{TNT-ALLOW TStrings}(Source).Strings[i])),
          TStrings{TNT-ALLOW TStrings}(Source).Objects[i]);
      GetAnsiStrings.Assign(UTF8Strings);
    end else if Source is TWideStrings then begin
      for i := 0 to TWideStrings(Source).Count - 1 do
        UTF8Strings.AddObject(WideStringToUTF8(TWideStrings(Source).Strings[i]),
          TWideStrings(Source).Objects[i]);
      GetAnsiStrings.Assign(UTF8Strings);
    end else
      GetAnsiStrings.Assign(Source);
  finally
    UTF8Strings.Free;
  end;
end;

function TTntStringGridStrings.GetAnsiStrings: TStrings{TNT-ALLOW TStrings};
begin
  Assert(Assigned(FGrid));
  if FIsCol then
    Result := TStringGrid{TNT-ALLOW TStringGrid}(FGrid).Cols[FColRowIndex]
  else
    Result := TStringGrid{TNT-ALLOW TStringGrid}(FGrid).Rows[FColRowIndex];
end;

procedure TTntStringGridStrings.Clear;
begin
  GetAnsiStrings.Clear;
end;

procedure TTntStringGridStrings.Delete(Index: Integer);
begin
  GetAnsiStrings.Delete(Index);
end;

function TTntStringGridStrings.GetCount: Integer;
begin
  Result := GetAnsiStrings.Count;
end;

function TTntStringGridStrings.Get(Index: Integer): WideString;
begin
  Result := UTF8ToWideString(GetAnsiStrings[Index]);
end;

procedure TTntStringGridStrings.Put(Index: Integer; const S: WideString);
begin
  GetAnsiStrings[Index] := WideStringToUTF8(S);
end;

procedure TTntStringGridStrings.Insert(Index: Integer; const S: WideString);
begin
  GetAnsiStrings.Insert(Index, WideStringToUTF8(S));
end;

function TTntStringGridStrings.Add(const S: WideString): Integer;
begin
  Result := GetAnsiStrings.Add(WideStringToUTF8(S));
end;

function TTntStringGridStrings.GetObject(Index: Integer): TObject;
begin
  Result := GetAnsiStrings.Objects[Index];
end;

procedure TTntStringGridStrings.PutObject(Index: Integer; AObject: TObject);
begin
  GetAnsiStrings.Objects[Index] := AObject;
end;

type TAccessStrings = class(TStrings{TNT-ALLOW TStrings});

procedure TTntStringGridStrings.SetUpdateState(Updating: Boolean);
begin
  TAccessStrings(GetAnsiStrings).SetUpdateState(Updating);
end;

constructor TTntStringGridStrings.Create(AGrid: TTntStringGrid; AIndex: Integer);
begin
  inherited Create;
  FGrid := AGrid;
  if AIndex > 0 then begin
    FIsCol := False;
    FColRowIndex := AIndex - 1;
  end else begin
    FIsCol := True;
    FColRowIndex := -AIndex - 1;
  end;
end;

{ _TTntInternalStringGrid }

procedure _TTntInternalStringGrid.SetEditText(ACol, ARow: Integer; const Value: string{TNT-ALLOW string});
begin
  if FSettingEditText then
    inherited
  else
    InternalSetEditText(ACol, ARow, Value);
end;

{ TTntStringGrid }

constructor TTntStringGrid.Create(AOwner: TComponent);
begin
  inherited;
  FCreatedRowStrings := TStringList{TNT-ALLOW TStringList}.Create;
  FCreatedRowStrings.Sorted := True;
  FCreatedRowStrings.Duplicates := dupError;
  FCreatedColStrings := TStringList{TNT-ALLOW TStringList}.Create;
  FCreatedColStrings.Sorted := True;
  FCreatedColStrings.Duplicates := dupError;
end;

destructor TTntStringGrid.Destroy;
var
  i: integer;
begin
  for i := FCreatedColStrings.Count - 1 downto 0 do
    FCreatedColStrings.Objects[i].Free;
  for i := FCreatedRowStrings.Count - 1 downto 0 do
    FCreatedRowStrings.Objects[i].Free;
  FreeAndNil(FCreatedColStrings);
  FreeAndNil(FCreatedRowStrings);
  inherited;
end;

function TTntStringGrid.CreateEditor: TInplaceEdit{TNT-ALLOW TInplaceEdit};
begin
  Result := TTntInplaceEdit.Create(Self);
end;

procedure TTntStringGrid.CreateWindowHandle(const Params: TCreateParams);
begin
  CreateUnicodeHandle(Self, Params, '');
end;

procedure TTntStringGrid.DefineProperties(Filer: TFiler);
begin
  inherited;
  DefineWideProperties(Filer, Self);
end;

function TTntStringGrid.IsHintStored: Boolean;
begin
  Result := TntControl_IsHintStored(Self);
end;

function TTntStringGrid.GetHint: WideString;
begin
  Result := TntControl_GetHint(Self)
end;

procedure TTntStringGrid.SetHint(const Value: WideString);
begin
  TntControl_SetHint(Self, Value);
end;

function TTntStringGrid.GetCells(ACol, ARow: Integer): WideString;
begin
  Result := UTF8ToWideString(inherited Cells[ACol, ARow])
end;

procedure TTntStringGrid.SetCells(ACol, ARow: Integer; const Value: WideString);
begin
  inherited Cells[ACol, ARow] := WideStringToUTF8(Value);
end;

function TTntStringGrid.FindGridStrings(IsCol: Boolean; ListIndex: Integer): TTntWideStrings;
var
  idx: integer;
  SrcStrings: TStrings{TNT-ALLOW TStrings};
  RCIndex: Integer;
begin
  if IsCol then
    SrcStrings := FCreatedColStrings
  else
    SrcStrings := FCreatedRowStrings;
  Assert(Assigned(SrcStrings));
  idx := SrcStrings.IndexOf(IntToStr(ListIndex));
  if idx <> -1 then
    Result := SrcStrings.Objects[idx] as TTntWideStrings
  else begin
    if IsCol then RCIndex := -ListIndex - 1 else RCIndex := ListIndex + 1;
    Result := TTntStringGridStrings.Create(Self, RCIndex);
    SrcStrings.AddObject(IntToStr(ListIndex), Result);
  end;
end;

function TTntStringGrid.GetCols(Index: Integer): TTntWideStrings;
begin
  Result := FindGridStrings(True, Index);
end;

function TTntStringGrid.GetRows(Index: Integer): TTntWideStrings;
begin
  Result := FindGridStrings(False, Index);
end;

procedure TTntStringGrid.SetCols(Index: Integer; const Value: TTntWideStrings);
begin
  FindGridStrings(True, Index).Assign(Value);
end;

procedure TTntStringGrid.SetRows(Index: Integer; const Value: TTntWideStrings);
begin
  FindGridStrings(False, Index).Assign(Value);
end;

procedure TTntStringGrid.DrawCell(ACol, ARow: Integer; ARect: TRect; AState: TGridDrawState);
var
  SaveDefaultDrawing: Boolean;
begin
  if DefaultDrawing then
    WideCanvasTextRect(Canvas, ARect, ARect.Left+2, ARect.Top+2, Cells[ACol, ARow]);
  SaveDefaultDrawing := DefaultDrawing;
  try
    DefaultDrawing := False;
    inherited DrawCell(ACol, ARow, ARect, AState);
  finally
    DefaultDrawing := SaveDefaultDrawing;
  end;
end;

function TTntStringGrid.GetEditText(ACol, ARow: Integer): WideString;
begin
  Result := Cells[ACol, ARow];
  if Assigned(FOnGetEditText) then FOnGetEditText(Self, ACol, ARow, Result);
end;

procedure TTntStringGrid.InternalSetEditText(ACol, ARow: Integer; const Value: string{TNT-ALLOW string});
begin
  if not FSettingEditText then
    SetEditText(ACol, ARow, TntControl_GetText(InplaceEditor));
end;

procedure TTntStringGrid.SetEditText(ACol, ARow: Integer; const Value: WideString);
begin
  FSettingEditText := True;
  try
    inherited SetEditText(ACol, ARow, WideStringToUTF8(Value));
  finally
    FSettingEditText := False;
  end;
  if Assigned(FOnSetEditText) then FOnSetEditText(Self, ACol, ARow, Value);
end;

procedure TTntStringGrid.WMChar(var Msg: TWMChar);
begin
  if (goEditing in Options)
  and (AnsiChar(Msg.CharCode) in [^H, #32..#255]) then begin
    RestoreWMCharMsg(TMessage(Msg));
    ShowEditorChar(WideChar(Msg.CharCode));
  end else
    inherited;
end;

procedure TTntStringGrid.ShowEditorChar(Ch: WideChar);
begin
  ShowEditor;
  if InplaceEditor <> nil then begin
    if Win32PlatformIsUnicode then
      PostMessageW(InplaceEditor.Handle, WM_CHAR, Word(Ch), 0)
    else
      PostMessageA(InplaceEditor.Handle, WM_CHAR, Word(Ch), 0);
  end;
end;

procedure TTntStringGrid.ActionChange(Sender: TObject; CheckDefaults: Boolean);
begin
  TntControl_BeforeInherited_ActionChange(Self, Sender, CheckDefaults);
  inherited;
end;

function TTntStringGrid.GetActionLinkClass: TControlActionLinkClass;
begin
  Result := TntControl_GetActionLinkClass(Self, inherited GetActionLinkClass);
end;

end.
