
{*******************************************************}
{    The Delphi Unicode Controls Project                }
{                                                       }
{      http://home.ccci.org/wolbrink                    }
{                                                       }
{ Copyright (c) 2002, Troy Wolbrink (wolbrink@ccci.org) }
{                                                       }
{*******************************************************}

unit TntStdCtrls;

interface

uses
  Windows, Messages, Classes, Controls, TntControls, StdCtrls, CheckLst, TntClasses;

{TNT-WARN TCustomEdit}
type
  TTntCustomEdit = class(TCustomEdit{TNT-ALLOW TCustomEdit})
  private
    function GetSelText: WideString; reintroduce;
    procedure SetSelText(const Value: WideString);
    function GetText: WideString;
    procedure SetText(const Value: WideString);
    function GetHint: WideString;
    procedure SetHint(const Value: WideString);
    function IsHintStored: Boolean;
  protected
    procedure CreateWindowHandle(const Params: TCreateParams); override;
    procedure DefineProperties(Filer: TFiler); override;
    function GetActionLinkClass: TControlActionLinkClass; override;
    procedure ActionChange(Sender: TObject; CheckDefaults: Boolean); override;
  public
    property SelText: WideString read GetSelText write SetSelText;
    property Text: WideString read GetText write SetText;
  published
    property Hint: WideString read GetHint write SetHint stored IsHintStored;
  end;

{TNT-WARN TEdit}
  TTntEdit = class(TTntCustomEdit)
  published
    property Anchors;
    property AutoSelect;
    property AutoSize;
{$IFNDEF VER130}
    property BevelEdges;
    property BevelInner;
    property BevelKind default bkNone;
    property BevelOuter;
{$ENDIF}
    property BiDiMode;
    property BorderStyle;
    property CharCase;
    property Color;
    property Constraints;
    property Ctl3D;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property Font;
    property HideSelection;
    property ImeMode;
    property ImeName;
    property MaxLength;
    property OEMConvert;
    property ParentBiDiMode;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PasswordChar;
    property PopupMenu;
    property ReadOnly;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Text;
    property Visible;
    property OnChange;
    property OnClick;
    property OnContextPopup;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDock;
    property OnStartDrag;
  end;

type
  TTntCustomMemo = class;

  TTntMemoStrings = class(TTntWideStrings)
  protected
    Memo: TCustomMemo{TNT-ALLOW TCustomMemo};
    FRichEditMode: Boolean;
    LineBreakStyle: TTntTextLineBreakStyle;
    function Get(Index: Integer): WideString; override;
    function GetCount: Integer; override;
    function GetTextStr: WideString; override;
    procedure SetTextStr(const Value: WideString); override;
    procedure Put(Index: Integer; const S: WideString); override;
    procedure SetUpdateState(Updating: Boolean); override;
  public
    constructor Create;
    procedure Clear; override;
    procedure Delete(Index: Integer); override;
    procedure Insert(Index: Integer; const S: WideString); override;
  end;

{TNT-WARN TCustomMemo}
  TTntCustomMemo = class(TCustomMemo{TNT-ALLOW TCustomMemo})
  private
    FLines: TTntWideStrings;
    procedure SetLines(const Value: TTntWideStrings);
    function GetSelText: WideString; reintroduce;
    procedure SetSelText(const Value: WideString);
    function GetText: WideString;
    procedure SetText(const Value: WideString);
    function GetHint: WideString;
    procedure SetHint(const Value: WideString);
    function IsHintStored: Boolean;
  protected
    procedure CreateWindowHandle(const Params: TCreateParams); override;
    procedure DefineProperties(Filer: TFiler); override;
    function GetActionLinkClass: TControlActionLinkClass; override;
    procedure ActionChange(Sender: TObject; CheckDefaults: Boolean); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property SelText: WideString read GetSelText write SetSelText;
    property Text: WideString read GetText write SetText;
    property Lines: TTntWideStrings read FLines write SetLines;
  published
    property Hint: WideString read GetHint write SetHint stored IsHintStored;
  end;

{TNT-WARN TMemo}
  TTntMemo = class(TTntCustomMemo)
  published
    property Align;
    property Alignment;
    property Anchors;
{$IFNDEF VER130}
    property BevelEdges;
    property BevelInner;
    property BevelKind default bkNone;
    property BevelOuter;
{$ENDIF}
    property BiDiMode;
    property BorderStyle;
    property Color;
    property Constraints;
    property Ctl3D;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property Font;
    property HideSelection;
    property ImeMode;
    property ImeName;
    property Lines;
    property MaxLength;
    property OEMConvert;
    property ParentBiDiMode;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ReadOnly;
    property ScrollBars;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property WantReturns;
    property WantTabs;
    property WordWrap;
    property OnChange;
    property OnClick;
    property OnContextPopup;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDock;
    property OnStartDrag;
  end;

  TTntComboBoxStrings = class(TTntWideStrings)
  protected
    function Get(Index: Integer): WideString; override;
    function GetCount: Integer; override;
    function GetObject(Index: Integer): TObject; override;
    procedure PutObject(Index: Integer; AObject: TObject); override;
    procedure SetUpdateState(Updating: Boolean); override;
  public
    ComboBox: TCustomComboBox{TNT-ALLOW TCustomComboBox};
    function Add(const S: WideString): Integer; override;
    procedure Clear; override;
    procedure Delete(Index: Integer); override;
    function IndexOf(const S: WideString): Integer; override;
    procedure Insert(Index: Integer; const S: WideString); override;
  end;

type
  ITntComboBox = interface
    ['{95B1B13F-96C4-4D36-98AF-2DF563D4B4A6}']
    procedure DoEditCharMsg(var Message: TWMChar);
    function GetItems: TTntWideStrings;
    procedure SetItems(const Value: TTntWideStrings);
    function GetSelLength: Integer;
    function GetSelStart: Integer;
    function GetSelText: WideString;
    procedure SetSelLength(Value: Integer);
    procedure SetSelStart(Value: Integer);
    procedure SetSelText(const Value: WideString);
    function GetText: WideString;
    procedure SetText(const Value: WideString);
    function GetItemIndex: Integer;
    procedure SetItemIndex(const Value: Integer);
    function GetAutoComplete_UniqueMatchOnly: Boolean;
    function GetAutoComplete_PreserveDataEntryCase: Boolean;
    //--
    property SelLength: Integer read GetSelLength write SetSelLength;
    property SelStart: Integer read GetSelStart write SetSelStart;
    property SelText: WideString read GetSelText write SetSelText;
    property Text: WideString read GetText write SetText;
    property Items: TTntWideStrings read GetItems write SetItems;
    property ItemIndex: Integer read GetItemIndex write SetItemIndex;
    property AutoComplete_UniqueMatchOnly: Boolean read GetAutoComplete_UniqueMatchOnly;
    property AutoComplete_PreserveDataEntryCase: Boolean read GetAutoComplete_PreserveDataEntryCase;
  end;

{$IFDEF VER150}
{ TD7PatchedComboBoxStrings }
type
  TD7PatchedComboBoxStrings = class(TCustomComboBoxStrings)
  protected
    function Get(Index: Integer): string{TNT-ALLOW string}; override;
  public
    function Add(const S: string{TNT-ALLOW string}): Integer; override;
    procedure Insert(Index: Integer; const S: string{TNT-ALLOW string}); override;
  end;
{$ENDIF}

{TNT-WARN TCustomComboBox}
type
  TTntCustomComboBox = class(TCustomComboBox{TNT-ALLOW TCustomComboBox},
    ITntComboBox, IWideCustomListControl)
  private
    FItems: TTntWideStrings;
    FSaveItems: TTntWideStrings;
    {$IFNDEF VER130}
    FFilter: WideString;
    FLastTime: Cardinal;
    {$ENDIF}
    function GetItems: TTntWideStrings;
    procedure SetItems(const Value: TTntWideStrings); reintroduce;
    function GetSelLength: Integer;
    procedure SetSelLength(Value: Integer);
    function GetSelStart: Integer;
    procedure SetSelStart(Value: Integer);
    function GetSelText: WideString;
    procedure SetSelText(const Value: WideString);
    function GetText: WideString;
    procedure SetText(const Value: WideString);
    function GetItemIndex: Integer; reintroduce;
    procedure SetItemIndex(const Value: Integer); reintroduce;
    procedure CNCommand(var Message: TWMCommand); message CN_COMMAND;
    function GetHint: WideString;
    procedure SetHint(const Value: WideString);
    function IsHintStored: Boolean;
    procedure WMChar(var Message: TWMChar); message WM_CHAR;
  protected
    procedure CreateWindowHandle(const Params: TCreateParams); override;
    procedure DefineProperties(Filer: TFiler); override;
    function GetActionLinkClass: TControlActionLinkClass; override;
    procedure ActionChange(Sender: TObject; CheckDefaults: Boolean); override;
    procedure DestroyWnd; override;
    function GetAutoComplete_UniqueMatchOnly: Boolean; dynamic;
    function GetAutoComplete_PreserveDataEntryCase: Boolean; dynamic;
    procedure DoEditCharMsg(var Message: TWMChar); virtual;
    procedure CreateWnd; override;
    procedure ComboWndProc(var Message: TMessage; ComboWnd: HWnd; ComboProc: Pointer); override;
    procedure DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState); override;
    procedure KeyPress(var Key: AnsiChar); override;
    {$IFDEF VER150}
    function GetItemsClass: TCustomComboBoxStringsClass; override;
    {$ENDIF}
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
{$IFNDEF VER130}
    procedure CopySelection(Destination: TCustomListControl); override;
{$ENDIF}
    procedure AddItem(const Item: WideString; AObject: TObject); reintroduce; virtual;
  public
    property SelText: WideString read GetSelText write SetSelText;
    property Text: WideString read GetText write SetText;
    property Items: TTntWideStrings read GetItems write SetItems;
  published
    property Hint: WideString read GetHint write SetHint stored IsHintStored;
  end;

{TNT-WARN TComboBox}
  TTntComboBox = class(TTntCustomComboBox)
  published
{$IFNDEF VER130}
    property AutoComplete default True;
    property AutoDropDown default False;
{$IFDEF VER150}
    property AutoCloseUp default False;
{$ENDIF}
    property BevelEdges;
    property BevelInner;
    property BevelKind default bkNone;
    property BevelOuter;
{$ENDIF}
    property Style; {Must be published before Items}
    property Anchors;
    property BiDiMode;
{$IFNDEF VER130}
    property CharCase;
{$ENDIF}
    property Color;
    property Constraints;
    property Ctl3D;
    property DragCursor;
    property DragKind;
    property DragMode;
    property DropDownCount;
    property Enabled;
    property Font;
    property ImeMode;
    property ImeName;
    property ItemHeight;
{$IFNDEF VER130}
    property ItemIndex default -1;
{$ENDIF}
    property MaxLength;
    property ParentBiDiMode;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property Sorted;
    property TabOrder;
    property TabStop;
    property Text;
    property Visible;
    property OnChange;
    property OnClick;
{$IFNDEF VER130}
    property OnCloseUp;
{$ENDIF}
    property OnContextPopup;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnDrawItem;
    property OnDropDown;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMeasureItem;
{$IFNDEF VER130}
    property OnSelect;
{$ENDIF}
    property OnStartDock;
    property OnStartDrag;
    property Items; { Must be published after OnMeasureItem }
  end;

  TAccessCustomListBox = class(TCustomListBox{TNT-ALLOW TCustomListBox});

  TTntListBoxStrings = class(TTntWideStrings)
  private
    FListBox: TAccessCustomListBox;
    function GetListBox: TCustomListBox{TNT-ALLOW TCustomListBox};
    procedure SetListBox(const Value: TCustomListBox{TNT-ALLOW TCustomListBox});
  protected
    procedure Put(Index: Integer; const S: WideString); override;
    function Get(Index: Integer): WideString; override;
    function GetCount: Integer; override;
    function GetObject(Index: Integer): TObject; override;
    procedure PutObject(Index: Integer; AObject: TObject); override;
    procedure SetUpdateState(Updating: Boolean); override;
  public
    function Add(const S: WideString): Integer; override;
    procedure Clear; override;
    procedure Delete(Index: Integer); override;
    procedure Exchange(Index1, Index2: Integer); override;
    function IndexOf(const S: WideString): Integer; override;
    procedure Insert(Index: Integer; const S: WideString); override;
    procedure Move(CurIndex, NewIndex: Integer); override;
    property ListBox: TCustomListBox{TNT-ALLOW TCustomListBox} read GetListBox write SetListBox;
  end;

{TNT-WARN TCustomListBox}
type
  TTntCustomListBox = class(TCustomListBox{TNT-ALLOW TCustomListBox}, IWideCustomListControl)
  private
    FItems: TTntWideStrings;
    FSaveItems: TTntWideStrings;
    FSaveTopIndex: Integer;
    FSaveItemIndex: Integer;
    procedure SetItems(const Value: TTntWideStrings);
    function GetHint: WideString;
    procedure SetHint(const Value: WideString);
    function IsHintStored: Boolean;
  protected
    procedure CreateWindowHandle(const Params: TCreateParams); override;
    procedure DefineProperties(Filer: TFiler); override;
    function GetActionLinkClass: TControlActionLinkClass; override;
    procedure ActionChange(Sender: TObject; CheckDefaults: Boolean); override;
    procedure CreateWnd; override;
    procedure DestroyWnd; override;
    procedure DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
{$IFNDEF VER130}
    procedure CopySelection(Destination: TCustomListControl); override;
{$ENDIF}
    procedure AddItem(const Item: WideString; AObject: TObject); reintroduce; virtual;
    property Items: TTntWideStrings read FItems write SetItems;
  published
    property Hint: WideString read GetHint write SetHint stored IsHintStored;
  end;

{TNT-WARN TListBox}
  TTntListBox = class(TTntCustomListBox)
  published
{$IFNDEF VER130}
    property Style;
    property AutoComplete;
{$ENDIF}
    property Align;
    property Anchors;
{$IFNDEF VER130}
    property BevelEdges;
    property BevelInner;
    property BevelKind default bkNone;
    property BevelOuter;
{$ENDIF}
    property BiDiMode;
    property BorderStyle;
    property Color;
    property Columns;
    property Constraints;
    property Ctl3D;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property ExtendedSelect;
    property Font;
    property ImeMode;
    property ImeName;
    property IntegralHeight;
    property ItemHeight;
    property Items;
    property MultiSelect;
    property ParentBiDiMode;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
{$IFNDEF VER130}
    property ScrollWidth;
{$ENDIF}
    property ShowHint;
    property Sorted;
{$IFDEF VER130}
    property Style;
{$ENDIF}
    property TabOrder;
    property TabStop;
    property TabWidth;
    property Visible;
    property OnClick;
    property OnContextPopup;
{$IFNDEF VER130}
    property OnData;
    property OnDataFind;
    property OnDataObject;
{$ENDIF}
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnDrawItem;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMeasureItem;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDock;
    property OnStartDrag;
  end;

{TNT-WARN TCustomLabel}
  TTntCustomLabel = class(TCustomLabel{TNT-ALLOW TCustomLabel})
  private
    function GetCaption: TWideCaption;
    procedure SetCaption(const Value: TWideCaption);
    function GetHint: WideString;
    procedure SetHint(const Value: WideString);
    function IsCaptionStored: Boolean;
    function IsHintStored: Boolean;
    procedure CMHintShow(var Message: TMessage); message CM_HINTSHOW;
    procedure CMDialogChar(var Message: TCMDialogChar); message CM_DIALOGCHAR;
  protected
    procedure DefineProperties(Filer: TFiler); override;
    function GetActionLinkClass: TControlActionLinkClass; override;
    procedure ActionChange(Sender: TObject; CheckDefaults: Boolean); override;
    function GetLabelText: WideString; reintroduce; virtual;
    procedure DoDrawText(var Rect: TRect; Flags: Longint); override;
    property Caption: TWideCaption read GetCaption write SetCaption stored IsCaptionStored;
  published
    property Hint: WideString read GetHint write SetHint stored IsHintStored;
  end;

{TNT-WARN TLabel}
  TTntLabel = class(TTntCustomLabel)
  published
    property Align;
    property Alignment;
    property Anchors;
    property AutoSize;
    property BiDiMode;
    property Caption;
    property Color {$IFDEF VER150} nodefault {$ENDIF};
    property Constraints;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property FocusControl;
    property Font;
    property ParentBiDiMode;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowAccelChar;
    property ShowHint;
    property Transparent;
    property Layout;
    property Visible;
    property WordWrap;
    property OnClick;
    property OnContextPopup;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDock;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
{$IFNDEF VER130}
    property OnMouseEnter;
    property OnMouseLeave;
{$ENDIF}
    property OnStartDock;
    property OnStartDrag;
  end;

{TNT-WARN TButton}
  TTntButton = class(TButton{TNT-ALLOW TButton})
  private
    function GetCaption: TWideCaption;
    procedure SetCaption(const Value: TWideCaption);
    function GetHint: WideString;
    procedure SetHint(const Value: WideString);
    function IsCaptionStored: Boolean;
    function IsHintStored: Boolean;
    procedure CMDialogChar(var Message: TCMDialogChar); message CM_DIALOGCHAR;
  protected
    procedure CreateWindowHandle(const Params: TCreateParams); override;
    procedure DefineProperties(Filer: TFiler); override;
    function GetActionLinkClass: TControlActionLinkClass; override;
    procedure ActionChange(Sender: TObject; CheckDefaults: Boolean); override;
  published
    property Caption: TWideCaption read GetCaption write SetCaption stored IsCaptionStored;
    property Hint: WideString read GetHint write SetHint stored IsHintStored;
  end;

{TNT-WARN TCustomCheckBox}
  TTntCustomCheckBox = class(TCustomCheckBox{TNT-ALLOW TCustomCheckBox})
  private
    function GetCaption: TWideCaption;
    procedure SetCaption(const Value: TWideCaption);
    function GetHint: WideString;
    procedure SetHint(const Value: WideString);
    function IsCaptionStored: Boolean;
    function IsHintStored: Boolean;
    procedure CMDialogChar(var Message: TCMDialogChar); message CM_DIALOGCHAR;
  protected
    procedure CreateWindowHandle(const Params: TCreateParams); override;
    procedure DefineProperties(Filer: TFiler); override;
    function GetActionLinkClass: TControlActionLinkClass; override;
    procedure ActionChange(Sender: TObject; CheckDefaults: Boolean); override;
  public
    property Caption: TWideCaption read GetCaption write SetCaption stored IsCaptionStored;
  published
    property Hint: WideString read GetHint write SetHint stored IsHintStored;
  end;

{TNT-WARN TCheckBox}
  TTntCheckBox = class(TTntCustomCheckBox)
  published
    property Action;
    property Alignment;
    property AllowGrayed;
    property Anchors;
    property BiDiMode;
    property Caption;
    property Checked;
    property Color {$IFDEF VER150} nodefault {$ENDIF};
    property Constraints;
    property Ctl3D;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property Font;
    property ParentBiDiMode;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property State;
    property TabOrder;
    property TabStop;
    property Visible;
{$IFDEF VER150}
    property WordWrap;
{$ENDIF}
    property OnClick;
    property OnContextPopup;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDock;
    property OnStartDrag;
  end;

{TNT-WARN TRadioButton}
  TTntRadioButton = class(TRadioButton{TNT-ALLOW TRadioButton})
  private
    function GetCaption: TWideCaption;
    procedure SetCaption(const Value: TWideCaption);
    function GetHint: WideString;
    procedure SetHint(const Value: WideString);
    function IsCaptionStored: Boolean;
    function IsHintStored: Boolean;
    procedure CMDialogChar(var Message: TCMDialogChar); message CM_DIALOGCHAR;
  protected
    procedure CreateWindowHandle(const Params: TCreateParams); override;
    procedure DefineProperties(Filer: TFiler); override;
    function GetActionLinkClass: TControlActionLinkClass; override;
    procedure ActionChange(Sender: TObject; CheckDefaults: Boolean); override;
  published
    property Caption: TWideCaption read GetCaption write SetCaption stored IsCaptionStored;
    property Hint: WideString read GetHint write SetHint stored IsHintStored;
  end;

{TNT-WARN TScrollBar}
  TTntScrollBar = class(TScrollBar{TNT-ALLOW TScrollBar})
  private
    function GetHint: WideString;
    procedure SetHint(const Value: WideString);
    function IsHintStored: Boolean;
  protected
    procedure CreateWindowHandle(const Params: TCreateParams); override;
    procedure DefineProperties(Filer: TFiler); override;
    function GetActionLinkClass: TControlActionLinkClass; override;
    procedure ActionChange(Sender: TObject; CheckDefaults: Boolean); override;
  published
    property Hint: WideString read GetHint write SetHint stored IsHintStored;
  end;

{TNT-WARN TCustomGroupBox}
  TTntCustomGroupBox = class(TCustomGroupBox{TNT-ALLOW TCustomGroupBox})
  private
    function GetCaption: TWideCaption;
    procedure SetCaption(const Value: TWideCaption);
    function GetHint: WideString;
    procedure SetHint(const Value: WideString);
    function IsCaptionStored: Boolean;
    function IsHintStored: Boolean;
    procedure CMDialogChar(var Message: TCMDialogChar); message CM_DIALOGCHAR;
  protected
    procedure Paint; override;
    procedure CreateWindowHandle(const Params: TCreateParams); override;
    procedure DefineProperties(Filer: TFiler); override;
    function GetActionLinkClass: TControlActionLinkClass; override;
    procedure ActionChange(Sender: TObject; CheckDefaults: Boolean); override;
    property Caption: TWideCaption read GetCaption write SetCaption stored IsCaptionStored;
  published
    property Hint: WideString read GetHint write SetHint stored IsHintStored;
  end;

{TNT-WARN TGroupBox}
  TTntGroupBox = class(TTntCustomGroupBox)
  published
    property Align;
    property Anchors;
    property BiDiMode;
    property Caption;
    property Color;
    property Constraints;
    property Ctl3D;
    property DockSite;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property Font;
{$IFDEF VER150}
    property ParentBackground default True;
{$ENDIF}
    property ParentBiDiMode;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnClick;
    property OnContextPopup;
    property OnDblClick;
    property OnDragDrop;
    property OnDockDrop;
    property OnDockOver;
    property OnDragOver;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnGetSiteInfo;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDock;
    property OnStartDrag;
    property OnUnDock;
  end;

{TNT-WARN TCustomStaticText}
  TTntCustomStaticText = class(TCustomStaticText{TNT-ALLOW TCustomStaticText})
  private
    function GetCaption: TWideCaption;
    procedure SetCaption(const Value: TWideCaption);
    function GetHint: WideString;
    procedure SetHint(const Value: WideString);
    function IsCaptionStored: Boolean;
    function IsHintStored: Boolean;
    procedure CMDialogChar(var Message: TCMDialogChar); message CM_DIALOGCHAR;
  protected
    procedure CreateWindowHandle(const Params: TCreateParams); override;
    procedure DefineProperties(Filer: TFiler); override;
    function GetActionLinkClass: TControlActionLinkClass; override;
    procedure ActionChange(Sender: TObject; CheckDefaults: Boolean); override;
    property Caption: TWideCaption read GetCaption write SetCaption stored IsCaptionStored;
  published
    property Hint: WideString read GetHint write SetHint stored IsHintStored;
  end;

{TNT-WARN TStaticText}
  TTntStaticText = class(TTntCustomStaticText)
  published
    property Align;
    property Alignment;
    property Anchors;
    property AutoSize;
    property BevelEdges;
    property BevelInner;
    property BevelKind default bkNone;
    property BevelOuter;
    property BiDiMode;
    property BorderStyle;
    property Caption;
    property Color {$IFDEF VER150} nodefault {$ENDIF};
    property Constraints;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property FocusControl;
    property Font;
    property ParentBiDiMode;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowAccelChar;
    property ShowHint;
    property TabOrder;
    property TabStop;
{$IFDEF VER150}
    property Transparent;
{$ENDIF}
    property Visible;
    property OnClick;
    property OnContextPopup;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDock;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDock;
    property OnStartDrag;
  end;

procedure TntCombo_AfterInherited_CreateWnd(Combo: TCustomComboBox{TNT-ALLOW TCustomComboBox}; var FSaveItems: TTntWideStrings);
procedure TntCombo_BeforeInherited_DestroyWnd(Combo: TCustomComboBox{TNT-ALLOW TCustomComboBox}; var FSaveItems: TTntWideStrings);
function TntCombo_ComboWndProc(Combo: TCustomComboBox{TNT-ALLOW TCustomComboBox};
  var Message: TMessage; ComboWnd: HWnd; ComboProc: Pointer): Boolean;
function TntCombo_CNCommand(Combo: TCustomComboBox{TNT-ALLOW TCustomComboBox}; var Message: TWMCommand): Boolean;
function TntCombo_GetSelText(Combo: TCustomComboBox{TNT-ALLOW TCustomComboBox}): WideString;
procedure TntCombo_SetSelText(Combo: TCustomComboBox{TNT-ALLOW TCustomComboBox}; const Value: WideString);
procedure TntCombo_BeforeKeyPress(Combo: TCustomComboBox{TNT-ALLOW TCustomComboBox}; var SaveAutoComplete: Boolean);
procedure TntCombo_AfterKeyPress(Combo: TCustomComboBox{TNT-ALLOW TCustomComboBox}; var SaveAutoComplete: Boolean);
procedure TntCombo_DropDown_PreserveSelection(Combo: TCustomComboBox{TNT-ALLOW TCustomComboBox});
procedure TntComboBox_AddItem(Items: TTntWideStrings; const Item: WideString; AObject: TObject);
{$IFNDEF VER130}
procedure TntComboBox_CopySelection(Items: TTntWideStrings; ItemIndex: Integer;
  Destination: TCustomListControl);
procedure TntCombo_AutoSearchKeyPress(Combo: TCustomComboBox{TNT-ALLOW TCustomComboBox}; var Message: TWMChar; var FFilter: WideString; var FLastTime: Cardinal);
procedure TntCombo_AutoCompleteKeyPress(Combo: TCustomComboBox{TNT-ALLOW TCustomComboBox}; var Message: TWMChar);
{$ENDIF}

procedure TntCustomEdit_CreateWindowHandle(Edit: TCustomEdit{TNT-ALLOW TCustomEdit}; const Params: TCreateParams);
function TntCustomEdit_GetSelText(Edit: TCustomEdit{TNT-ALLOW TCustomEdit}): WideString;
procedure TntCustomEdit_SetSelText(Edit: TCustomEdit{TNT-ALLOW TCustomEdit}; const Value: WideString);

function TntMemo_LineStart(Handle: THandle; Index: Integer): Integer;
function TntMemo_LineLength(Handle: THandle; Index: Integer): Integer;

procedure TntListBox_AfterInherited_CreateWnd(ListBox: TCustomListBox{TNT-ALLOW TCustomListBox};
  var FSaveItems: TTntWideStrings; FItems: TTntWideStrings; FSaveTopIndex, FSaveItemIndex: Integer);
procedure TntListBox_BeforeInherited_DestroyWnd(ListBox: TCustomListBox{TNT-ALLOW TCustomListBox};
  var FSaveItems: TTntWideStrings; const FItems: TTntWideStrings; var FSaveTopIndex, FSaveItemIndex: Integer);
procedure TntListBox_DrawItem_Text(ListBox: TCustomListBox{TNT-ALLOW TCustomListBox}; Items: TTntWideStrings; Index: Integer; Rect: TRect);
procedure TntListBox_AddItem(Items: TTntWideStrings; const Item: WideString; AObject: TObject);
{$IFNDEF VER130}
procedure TntListBox_CopySelection(ListBox: TCustomListbox{TNT-ALLOW TCustomListbox};
  Items: TTntWideStrings; Destination: TCustomListControl);
{$ENDIF}

procedure TntButton_CMDialogChar(Button: TButton{TNT-ALLOW TButton}; var Message: TCMDialogChar);

implementation

uses Forms, SysUtils, Consts, RichEdit, Graphics, ComStrs, Dialogs,
     {$IFNDEF VER130} RTLConsts, {$ENDIF} {$IFDEF VER150} Themes, {$ENDIF}
     TntForms, TntGraphics, TntWideStrPropHelper, TntActnList;

{ TTntCustomEdit }

procedure TntCustomEdit_CreateWindowHandle(Edit: TCustomEdit{TNT-ALLOW TCustomEdit}; const Params: TCreateParams);
var
  P: TCreateParams;
begin
  if SysLocale.FarEast
  and (not Win32PlatformIsUnicode)
  and ((Params.Style and ES_READONLY) <> 0) then begin
    // Work around Far East Win95 API/IME bug.
    P := Params;
    P.Style := P.Style and (not ES_READONLY);
    CreateUnicodeHandle(Edit, P, 'EDIT');
    if Edit.HandleAllocated then
      SendMessage(Edit.Handle, EM_SETREADONLY, Ord(True), 0);
  end else
    CreateUnicodeHandle(Edit, Params, 'EDIT');
end;

function TntCustomEdit_GetSelText(Edit: TCustomEdit{TNT-ALLOW TCustomEdit}): WideString;
begin
  if (not IsWindowUnicode(Edit.Handle)) then
    Result := Edit.SelText
  else
    Result := Copy(TntControl_GetText(Edit), Edit.SelStart + 1, Edit.SelLength);
end;

procedure TntCustomEdit_SetSelText(Edit: TCustomEdit{TNT-ALLOW TCustomEdit}; const Value: WideString);
begin
  if (not IsWindowUnicode(Edit.Handle)) then
    Edit.SelText := Value
  else
    SendMessageW(Edit.Handle, EM_REPLACESEL, 0, Longint(PWideChar(Value)));
end;

procedure TTntCustomEdit.CreateWindowHandle(const Params: TCreateParams);
begin
  TntCustomEdit_CreateWindowHandle(Self, Params);
end;

procedure TTntCustomEdit.DefineProperties(Filer: TFiler);
begin
  inherited;
  DefineWideProperties(Filer, Self);
end;

function TTntCustomEdit.GetSelText: WideString;
begin
  Result := TntCustomEdit_GetSelText(Self);
end;

procedure TTntCustomEdit.SetSelText(const Value: WideString);
begin
  TntCustomEdit_SetSelText(Self, Value);
end;

function TTntCustomEdit.GetText: WideString;
begin
  Result := TntControl_GetText(Self);
end;

procedure TTntCustomEdit.SetText(const Value: WideString);
begin
  TntControl_SetText(Self, Value);
end;

function TTntCustomEdit.IsHintStored: Boolean;
begin
  Result := TntControl_IsHintStored(Self);
end;

function TTntCustomEdit.GetHint: WideString;
begin
  Result := TntControl_GetHint(Self)
end;

procedure TTntCustomEdit.SetHint(const Value: WideString);
begin
  TntControl_SetHint(Self, Value);
end;

procedure TTntCustomEdit.ActionChange(Sender: TObject; CheckDefaults: Boolean);
begin
  TntControl_BeforeInherited_ActionChange(Self, Sender, CheckDefaults);
  inherited;
end;

function TTntCustomEdit.GetActionLinkClass: TControlActionLinkClass;
begin
  Result := TntControl_GetActionLinkClass(Self, inherited GetActionLinkClass);
end;

{ TTntMemoStrings }

constructor TTntMemoStrings.Create;
begin
  inherited;
  LineBreakStyle := tlbsCRLF;
end;

function TTntMemoStrings.GetCount: Integer;
begin
  Result := Memo.Lines.Count;
end;

function TntMemo_LineStart(Handle: THandle; Index: Integer): Integer;
begin
  Assert(Win32PlatformIsUnicode);
  Result := SendMessageW(Handle, EM_LINEINDEX, Index, 0);
end;

function TntMemo_LineLength(Handle: THandle; Index: Integer): Integer;
var
  SelStart: Integer;
begin
  Assert(Win32PlatformIsUnicode);
  SelStart := TntMemo_LineStart(Handle, Index);
  if SelStart < 0 then
    Result := 0
  else
    Result := SendMessageW(Handle, EM_LINELENGTH, SelStart, 0);
end;

function TTntMemoStrings.Get(Index: Integer): WideString;
var
  Len: Integer;
begin
  if (not IsWindowUnicode(Memo.Handle)) then
    Result := Memo.Lines[Index]
  else begin
    SetLength(Result, TntMemo_LineLength(Memo.Handle, Index));
    if Length(Result) > 0 then begin
      Word((PWideChar(Result))^) := Length(Result);
      Len := SendMessageW(Memo.Handle, EM_GETLINE, Index, Longint(PWideChar(Result)));
      SetLength(Result, Len);
    end;
  end;
end;

procedure TTntMemoStrings.Put(Index: Integer; const S: WideString);
var
  SelStart: Integer;
begin
  if (not IsWindowUnicode(Memo.Handle)) then
    Memo.Lines[Index] := S
  else begin
    SelStart := TntMemo_LineStart(Memo.Handle, Index);
    if SelStart >= 0 then
    begin
      SendMessageW(Memo.Handle, EM_SETSEL, SelStart, SelStart + TntMemo_LineLength(Memo.Handle, Index));
      SendMessageW(Memo.Handle, EM_REPLACESEL, 0, Longint(PWideChar(S)));
    end;
  end;
end;

procedure TTntMemoStrings.Insert(Index: Integer; const S: Widestring);

  function RichEditSelStart: Integer;
  var
    CharRange: TCharRange;
  begin
    SendMessage(Memo.Handle, EM_EXGETSEL, 0, Longint(@CharRange));
    Result := CharRange.cpMin;
  end;

var
  SelStart, LineLen: Integer;
  Line: WideString;
begin
  if (not IsWindowUnicode(Memo.Handle)) then
    Memo.Lines.Insert(Index, S)
  else begin
    if Index >= 0 then
    begin
      SelStart := TntMemo_LineStart(Memo.Handle, Index);
      if SelStart >= 0 then
        Line := S + CRLF
      else begin
        SelStart := TntMemo_LineStart(Memo.Handle, Index - 1);
        LineLen := TntMemo_LineLength(Memo.Handle, Index - 1);
        if LineLen = 0 then
          Exit;
        Inc(SelStart, LineLen);
        Line := CRLF + s;
      end;
      if (FRichEditMode)
      and (LineBreakStyle <> tlbsCRLF) then
        Line := TntAdjustLineBreaks(Line, LineBreakStyle);
      SendMessageW(Memo.Handle, EM_SETSEL, SelStart, SelStart);
      SendMessageW(Memo.Handle, EM_REPLACESEL, 0, Longint(PWideChar(Line)));
      if (FRichEditMode)
      and (RichEditSelStart <> (SelStart + Length(Line))) then
        raise EOutOfResources.Create(sRichEditInsertError);
    end;
  end;
end;

procedure TTntMemoStrings.Delete(Index: Integer);
begin
  Memo.Lines.Delete(Index);
end;

procedure TTntMemoStrings.Clear;
begin
  Memo.Lines.Clear;
end;

type TAccessStrings = class(TStrings{TNT-ALLOW TStrings});

procedure TTntMemoStrings.SetUpdateState(Updating: Boolean);
begin
  TAccessStrings(Memo.Lines).SetUpdateState(Updating);
end;

function TTntMemoStrings.GetTextStr: WideString;
begin
  if (not FRichEditMode) then
    Result := TntControl_GetText(Memo)
  else
    Result := inherited GetTextStr;
end;

procedure TTntMemoStrings.SetTextStr(const Value: WideString);
begin
  TntControl_SetText(Memo, TntAdjustLineBreaks(Value, LineBreakStyle));
end;

{ TTntCustomMemo }

constructor TTntCustomMemo.Create(AOwner: TComponent);
begin
  inherited;
  FLines := TTntMemoStrings.Create;
  TTntMemoStrings(FLines).Memo := Self;
end;

destructor TTntCustomMemo.Destroy;
begin
  FreeAndNil(FLines);
  inherited;
end;

procedure TTntCustomMemo.SetLines(const Value: TTntWideStrings);
begin
  FLines.Assign(Value);
end;

procedure TTntCustomMemo.CreateWindowHandle(const Params: TCreateParams);
begin
  TntCustomEdit_CreateWindowHandle(Self, Params);
end;

procedure TTntCustomMemo.DefineProperties(Filer: TFiler);
begin
  inherited;
  DefineWideProperties(Filer, Self);
end;

function TTntCustomMemo.GetSelText: WideString;
begin
  Result := TntCustomEdit_GetSelText(Self);
end;

procedure TTntCustomMemo.SetSelText(const Value: WideString);
begin
  TntCustomEdit_SetSelText(Self, Value);
end;

function TTntCustomMemo.GetText: WideString;
begin
  Result := TntControl_GetText(Self);
end;

procedure TTntCustomMemo.SetText(const Value: WideString);
begin
  TntControl_SetText(Self, Value);
end;

function TTntCustomMemo.IsHintStored: Boolean;
begin
  Result := TntControl_IsHintStored(Self);
end;

function TTntCustomMemo.GetHint: WideString;
begin
  Result := TntControl_GetHint(Self)
end;

procedure TTntCustomMemo.SetHint(const Value: WideString);
begin
  TntControl_SetHint(Self, Value);
end;

procedure TTntCustomMemo.ActionChange(Sender: TObject; CheckDefaults: Boolean);
begin
  TntControl_BeforeInherited_ActionChange(Self, Sender, CheckDefaults);
  inherited;
end;

function TTntCustomMemo.GetActionLinkClass: TControlActionLinkClass;
begin
  Result := TntControl_GetActionLinkClass(Self, inherited GetActionLinkClass);
end;

{$IFDEF VER150}
function TD7PatchedComboBoxStrings.Get(Index: Integer): string{TNT-ALLOW string};
var
  Len: Integer;
begin
  Len := SendMessage(ComboBox.Handle, CB_GETLBTEXTLEN, Index, 0);
  if Len > 0 then
  begin
    SetLength(Result, Len);
    SendMessage(ComboBox.Handle, CB_GETLBTEXT, Index, Longint(PChar{TNT-ALLOW PChar}(Result)));
  end
  else
    SetLength(Result, 0);
end;

function TD7PatchedComboBoxStrings.Add(const S: string{TNT-ALLOW string}): Integer;
begin
  Result := SendMessage(ComboBox.Handle, CB_ADDSTRING, 0, Longint(PChar{TNT-ALLOW PChar}(S)));
  if Result < 0 then
    raise EOutOfResources.Create(SInsertLineError);
end;

procedure TD7PatchedComboBoxStrings.Insert(Index: Integer; const S: string{TNT-ALLOW string});
begin
  if SendMessage(ComboBox.Handle, CB_INSERTSTRING, Index,
    Longint(PChar{TNT-ALLOW PChar}(S))) < 0 then
    raise EOutOfResources.Create(SInsertLineError);
end;
{$ENDIF}

{ TTntComboBoxStrings }

function TTntComboBoxStrings.GetCount: Integer;
begin
  Result := ComboBox.Items.Count;
end;

function TTntComboBoxStrings.Get(Index: Integer): WideString;
var
  Len: Integer;
begin
  if (not IsWindowUnicode(ComboBox.Handle)) then
    Result := ComboBox.Items[Index]
  else begin
    Len := SendMessageW(ComboBox.Handle, CB_GETLBTEXTLEN, Index, 0);
    if Len = CB_ERR then
      Result := ''
    else begin
      SetLength(Result, Len + 1);
      Len := SendMessageW(ComboBox.Handle, CB_GETLBTEXT, Index, Longint(PWideChar(Result)));
      if Len = CB_ERR then
        Result := ''
       else
        Result := PWideChar(Result);
    end;
  end;
end;

function TTntComboBoxStrings.GetObject(Index: Integer): TObject;
begin
  Result := ComboBox.Items.Objects[Index];
end;

procedure TTntComboBoxStrings.PutObject(Index: Integer; AObject: TObject);
begin
  ComboBox.Items.Objects[Index] := AObject;
end;

function TTntComboBoxStrings.Add(const S: WideString): Integer;
begin
  if (not IsWindowUnicode(ComboBox.Handle)) then
    Result := ComboBox.Items.Add(S)
  else begin
    Result := SendMessageW(ComboBox.Handle, CB_ADDSTRING, 0, Longint(PWideChar(S)));
    if Result < 0 then
      raise EOutOfResources.Create(SInsertLineError);
  end;
end;

procedure TTntComboBoxStrings.Insert(Index: Integer; const S: WideString);
begin
  if (not IsWindowUnicode(ComboBox.Handle)) then
    ComboBox.Items.Insert(Index, S)
  else begin
    if SendMessageW(ComboBox.Handle, CB_INSERTSTRING, Index, Longint(PWideChar(S))) < 0 then
      raise EOutOfResources.Create(SInsertLineError);
  end;
end;

procedure TTntComboBoxStrings.Delete(Index: Integer);
begin
  ComboBox.Items.Delete(Index);
end;

procedure TTntComboBoxStrings.Clear;
var
  S: WideString;
begin
  S := TntControl_GetText(ComboBox);
  SendMessage(ComboBox.Handle, CB_RESETCONTENT, 0, 0);
  TntControl_SetText(ComboBox, S);
  ComboBox.Update;
end;

procedure TTntComboBoxStrings.SetUpdateState(Updating: Boolean);
begin
  TAccessStrings(ComboBox.Items).SetUpdateState(Updating);
end;

function TTntComboBoxStrings.IndexOf(const S: WideString): Integer;
begin
  if (not IsWindowUnicode(ComboBox.Handle)) then
    Result := ComboBox.Items.IndexOf(S)
  else
    Result := SendMessageW(ComboBox.Handle, CB_FINDSTRINGEXACT, -1, LongInt(PWideChar(S)));
end;

{ TTntCustomComboBox }

type TAccessCustomComboBox = class(TCustomComboBox{TNT-ALLOW TCustomComboBox});

procedure TntCombo_AfterInherited_CreateWnd(Combo: TCustomComboBox{TNT-ALLOW TCustomComboBox}; var FSaveItems: TTntWideStrings);
var
  FSaveItemIndex: Integer;
  TntCombo: ITntComboBox;
begin
  Combo.GetInterface(ITntComboBox, TntCombo);
  Assert(TntCombo <> nil);
  with TAccessCustomComboBox(Combo), TntCombo do begin
    if Win32PlatformIsUnicode then begin
      if ListHandle <> 0 then
        SetWindowLongW(ListHandle, GWL_WNDPROC, GetWindowLong(ListHandle, GWL_WNDPROC));
      SetWindowLongW(EditHandle, GWL_WNDPROC, GetWindowLong(EditHandle, GWL_WNDPROC));
      if FSaveItems <> nil then
      begin
        FSaveItemIndex := ItemIndex;
        Items.Assign(FSaveItems);
        FreeAndNil(FSaveItems);
        TntControl_SetText(Combo, TntControl_GetStoredText(Combo, TAccessCustomComboBox(Combo).Text));
        if FSaveItemIndex <> -1 then
        begin
          if Items.Count < FSaveItemIndex then FSaveItemIndex := Items.Count;
          SendMessage(Handle, CB_SETCURSEL, FSaveItemIndex, 0);
        end;
      end;
    end;
  end;
end;

procedure TntCombo_BeforeInherited_DestroyWnd(Combo: TCustomComboBox{TNT-ALLOW TCustomComboBox}; var FSaveItems: TTntWideStrings);
var
  TntCombo: ITntComboBox;
begin
  Combo.GetInterface(ITntComboBox, TntCombo);
  Assert(TntCombo <> nil);
  with TAccessCustomComboBox(Combo), TntCombo do begin
    if (Win32PlatformIsUnicode) and (Items.Count > 0) and (not (csDestroying in ComponentState))
    then begin
      FSaveItems := TTntWideStringList.Create;
      FSaveItems.Assign(Items);
    end;
  end;
end;

function TntCombo_ComboWndProc(Combo: TCustomComboBox{TNT-ALLOW TCustomComboBox};
  var Message: TMessage; ComboWnd: HWnd; ComboProc: Pointer): Boolean;

  procedure CallDefaultWindowProc;
  begin
    with Message do begin { call default wnd proc }
      if IsWindowUnicode(ComboWnd) then
        Result := CallWindowProcW(ComboProc, ComboWnd, Msg, WParam, LParam)
      else
        Result := CallWindowProcA(ComboProc, ComboWnd, Msg, WParam, LParam);
    end;
  end;

var
  TntCombo: ITntComboBox;

  function DoWideKeyPress(Message: TWMChar): Boolean;
  begin
    TntCombo.DoEditCharMsg(Message);
    Result := (Message.CharCode = 0);
  end;

begin
  Combo.GetInterface(ITntComboBox, TntCombo);
  Assert(TntCombo <> nil);
  Result := False;
  try
    with TAccessCustomComboBox(Combo), TntCombo do begin
      if (Message.Msg = WM_CHAR) then begin
        // WM_CHAR
        Result := True;
        if IsWindowUnicode(ComboWnd) then
          MakeWMCharMsgSafeForAnsi(Message);
        try
          if DoKeyPress(TWMKey(Message)) then Exit;
          if DoWideKeyPress(TWMKey(Message)) then Exit;
        finally
          if IsWindowUnicode(ComboWnd) then
            RestoreWMCharMsg(Message);
        end;
        with TWMKey(Message) do begin
          if ((CharCode = VK_RETURN) or (CharCode = VK_ESCAPE)) and DroppedDown then begin
            DroppedDown := False;
            Exit;
          end;
        end;
        CallDefaultWindowProc;
      end else if (IsWindowUnicode(ComboWnd)) then begin
        // UNICODE
        if IsTextMessage(Message.Msg) then begin
          // WM_*TEXT*
          Result := True;
          CallDefaultWindowProc;
        end else if (Message.Msg = WM_IME_CHAR) then begin
          // WM_IME_CHAR
          Result := True;
          with Message do { convert to WM_CHAR }
            Result := SendMessageW(ComboWnd, WM_CHAR, WParam, LParam);
        end;
      end;
    end;
  except
    Application.HandleException(Combo);
  end;
end;

function TntCombo_CNCommand(Combo: TCustomComboBox{TNT-ALLOW TCustomComboBox}; var Message: TWMCommand): Boolean;
var
  TntCombo: ITntComboBox;
begin
  Combo.GetInterface(ITntComboBox, TntCombo);
  Assert(TntCombo <> nil);
  with TAccessCustomComboBox(Combo), TntCombo do begin
    Result := False;
    if Message.NotifyCode = CBN_SELCHANGE then begin
      Result := True;
      Text := Items[ItemIndex];
      Click;
      {$IFDEF VER130}
      Change;
      {$ELSE}
      Select;
      {$ENDIF}
     end;
  end;
end;

function TntCombo_GetSelText(Combo: TCustomComboBox{TNT-ALLOW TCustomComboBox}): WideString;
var
  TntCombo: ITntComboBox;
begin
  Combo.GetInterface(ITntComboBox, TntCombo);
  Assert(TntCombo <> nil);
  with TAccessCustomComboBox(Combo), TntCombo do begin
    if (not IsWindowUnicode(Handle)) then
      Result := Combo.SelText
    else begin
      Result := '';
      if Style < csDropDownList then
        Result := Copy(Text, SelStart + 1, SelLength);
    end;
  end;
end;

procedure TntCombo_SetSelText(Combo: TCustomComboBox{TNT-ALLOW TCustomComboBox}; const Value: WideString);
var
  TntCombo: ITntComboBox;
begin
  Combo.GetInterface(ITntComboBox, TntCombo);
  Assert(TntCombo <> nil);
  with TAccessCustomComboBox(Combo), TntCombo do begin
    if (not IsWindowUnicode(Handle)) then
      Combo.SelText := Value
    else begin
      if Style < csDropDownList then
      begin
        HandleNeeded;
        SendMessageW(EditHandle, EM_REPLACESEL, 0, Longint(PWideChar(Value)));
      end;
    end;
  end;
end;

procedure TntCombo_BeforeKeyPress(Combo: TCustomComboBox{TNT-ALLOW TCustomComboBox}; var SaveAutoComplete: Boolean);
begin
  {$IFNDEF VER130}
  SaveAutoComplete := TAccessCustomComboBox(Combo).AutoComplete;
  TAccessCustomComboBox(Combo).AutoComplete := False;
  {$ENDIF}
end;

procedure TntCombo_AfterKeyPress(Combo: TCustomComboBox{TNT-ALLOW TCustomComboBox}; var SaveAutoComplete: Boolean);
begin
  {$IFNDEF VER130}
  TAccessCustomComboBox(Combo).AutoComplete := SaveAutoComplete;
  {$ENDIF}
end;

procedure TntCombo_DropDown_PreserveSelection(Combo: TCustomComboBox{TNT-ALLOW TCustomComboBox});
var
  TntCombo: ITntComboBox;
  OldSelStart, OldSelLength: Integer;
  OldText: WideString;
begin
  Combo.GetInterface(ITntComboBox, TntCombo);
  Assert(TntCombo <> nil);
  with TAccessCustomComboBox(Combo), TntCombo do begin
    OldText := Text;
    OldSelStart := SelStart;
    OldSelLength := SelLength;
    DroppedDown := True;
    Text := OldText;
    SelStart := OldSelStart;
    SelLength := OldSelLength;
  end;
end;

procedure TntComboBox_AddItem(Items: TTntWideStrings; const Item: WideString; AObject: TObject);
begin
  Items.AddObject(Item, AObject);
end;

{$IFNDEF VER130}
procedure TntComboBox_CopySelection(Items: TTntWideStrings; ItemIndex: Integer;
  Destination: TCustomListControl);
begin
  if ItemIndex <> -1 then
    WideListControl_AddItem(Destination, Items[ItemIndex], Items.Objects[ItemIndex]);
end;

function TntCombo_FindString(hWnd: THandle; StartPos: Integer; const Text: WideString): Integer;
begin
  if IsWindowUnicode(hWnd) then
    Result := SendMessageW(hWnd, CB_FINDSTRING, StartPos, Integer(PWideChar(Text)))
  else
    Result := SendMessageA(hWnd, CB_FINDSTRING, StartPos, Integer(PAnsiChar(AnsiString(Text))))
end;

function TntCombo_FindUniqueString(hWnd: THandle; StartPos: Integer; const Text: WideString): Integer;
var
  Match_1, Match_2: Integer;
begin
  Result := CB_ERR;
  Match_1 := TntCombo_FindString(hWnd, -1, Text);
  if Match_1 <> CB_ERR then begin
    Match_2 := TntCombo_FindString(hWnd, Match_1, Text);
    if Match_2 = Match_1 then
      Result := Match_1;
  end;
end;

function TntCombo_AutoSelect(Combo: TCustomComboBox{TNT-ALLOW TCustomComboBox}; const SearchText: WideString;
  UniqueMatchOnly: Boolean; UseDataEntryCase: Boolean): Boolean;
var
  Idx: Integer;
  ValueChange: Boolean;
  TntCombo: ITntComboBox;
begin
  Combo.GetInterface(ITntComboBox, TntCombo);
  Assert(TntCombo <> nil);
  with TAccessCustomComboBox(Combo), TntCombo do begin
    if UniqueMatchOnly then
      Idx := TntCombo_FindUniqueString(Handle, -1, SearchText)
    else
      Idx := TntCombo_FindString(Handle, -1, SearchText);
    Result := (Idx <> CB_ERR);
    if Result then begin
      ValueChange := Idx <> ItemIndex;
      {$IFDEF VER150}
      // auto-closeup
      if AutoCloseUp and (Items.IndexOf(SearchText) <> -1) then
        DroppedDown := False;
      {$ENDIF}
      // select item
      ItemIndex := Idx;
      // update edit
      if (Style in [csDropDown, csSimple]) then begin
        if UseDataEntryCase then begin
          // preserve case of characters as they are entered
          Text := SearchText + Copy(Items[ItemIndex], Length(SearchText) + 1, MaxInt);
        end;
        // select the rest of the string
        SelStart := Length(SearchText);
        SelLength := Length(Text) - SelStart;
      end;
      // notify events
      if ValueChange then begin
        Click;
        Select;
      end;
    end;
  end;
end;

procedure TntCombo_AutoSearchKeyPress(Combo: TCustomComboBox{TNT-ALLOW TCustomComboBox}; var Message: TWMChar;
  var FFilter: WideString; var FLastTime: Cardinal);
var
  TntCombo: ITntComboBox;
  Key: WideChar;
begin
  Combo.GetInterface(ITntComboBox, TntCombo);
  Assert(TntCombo <> nil);
  with TAccessCustomComboBox(Combo), TntCombo do begin
    Assert(not (Style in [csSimple, csDropDown]), 'Internal Error: TntCombo_AutoSearchKeyPress is not for csSimple and csDropDown style combo boxes.');
    if not AutoComplete then exit;
    Key := GetWideCharFromWMCharMsg(Message);
    try
      case Ord(Key) of
        VK_ESCAPE:
          exit;
        VK_TAB:
          if AutoDropDown and DroppedDown then
            DroppedDown := False;
        VK_BACK:
          Delete(FFilter, Length(FFilter), 1);
        else begin
          if AutoDropDown and (not DroppedDown) then
            Combo.DroppedDown := True;
          // reset FFilter if it's been too long (0.5 sec)
          if GetTickCount - FLastTime >= 500 then
            FFilter := '';
          FLastTime := GetTickCount;
          // if AutoSelect works, remember new FFilter
          if TntCombo_AutoSelect(Combo, FFilter + Key, False, True) then begin
            FFilter := FFilter + Key;
            Key := #0;
          end;
        end;
      end;
    finally
      SetWideCharForWMCharMsg(Message, Key);
    end;
  end;
end;

procedure TntCombo_AutoCompleteKeyPress(Combo: TCustomComboBox{TNT-ALLOW TCustomComboBox}; var Message: TWMChar);
var
  TntCombo: ITntComboBox;
  Key: WideChar;
begin
  Combo.GetInterface(ITntComboBox, TntCombo);
  Assert(TntCombo <> nil);
  with TAccessCustomComboBox(Combo), TntCombo do begin
    Assert(Style in [csSimple, csDropDown], 'Internal Error: TntCombo_AutoCompleteKeyPress is only for csSimple and csDropDown style combo boxes.');
    if not AutoComplete then exit;
    Key := GetWideCharFromWMCharMsg(Message);
    try
      case Ord(Key) of
        VK_ESCAPE:
          exit;
        VK_TAB:
          if AutoDropDown and DroppedDown then
            DroppedDown := False;
        VK_BACK:
          exit;
        else begin
          if AutoDropDown and (not DroppedDown) then
            TntCombo_DropDown_PreserveSelection(Combo);
          // AutoComplete only if the selection is at the very end
          if ((SelStart + SelLength) = Length(Text)) then begin
            Text := Copy(Text, 1, SelStart) + Key;
            Change;
            SelStart := Length(Text);
            TntCombo_AutoSelect(Combo, Text, AutoComplete_UniqueMatchOnly, AutoComplete_PreserveDataEntryCase);
            Key := #0;
          end;
        end;
      end;
    finally
      SetWideCharForWMCharMsg(Message, Key);
    end;
  end;
end;
{$ENDIF}

//--
constructor TTntCustomComboBox.Create(AOwner: TComponent);
begin
  inherited;
  FItems := TTntComboBoxStrings.Create;
  TTntComboBoxStrings(FItems).ComboBox := Self;
end;

destructor TTntCustomComboBox.Destroy;
begin
  FreeAndNil(FItems);
  FreeAndNil(FSaveItems);
  inherited;
end;

procedure TTntCustomComboBox.CreateWindowHandle(const Params: TCreateParams);
begin
  CreateUnicodeHandle(Self, Params, 'COMBOBOX');
end;

procedure TTntCustomComboBox.DefineProperties(Filer: TFiler);
begin
  inherited;
  DefineWideProperties(Filer, Self);
end;

procedure TTntCustomComboBox.CreateWnd;
begin
  inherited;
  TntCombo_AfterInherited_CreateWnd(Self, FSaveItems);
end;

procedure TTntCustomComboBox.DestroyWnd;
begin
  TntCombo_BeforeInherited_DestroyWnd(Self, FSaveItems);
  inherited;
end;

procedure TTntCustomComboBox.ComboWndProc(var Message: TMessage; ComboWnd: HWnd; ComboProc: Pointer);
begin
  if not TntCombo_ComboWndProc(Self, Message, ComboWnd, ComboProc) then
    inherited;
end;

procedure TTntCustomComboBox.KeyPress(var Key: AnsiChar);
var
  SaveAutoComplete: Boolean;
begin
  TntCombo_BeforeKeyPress(Self, SaveAutoComplete);
  try
    inherited;
  finally
    TntCombo_AfterKeyPress(Self, SaveAutoComplete);
  end;
end;

procedure TTntCustomComboBox.DoEditCharMsg(var Message: TWMChar);
begin
  {$IFNDEF VER130}
  TntCombo_AutoCompleteKeyPress(Self, Message);
  {$ENDIF}
end;

procedure TTntCustomComboBox.WMChar(var Message: TWMChar);
begin
  {$IFNDEF VER130}
  TntCombo_AutoSearchKeyPress(Self, Message, FFilter, FLastTime);
  {$ENDIF}
  inherited;
end;

procedure TTntCustomComboBox.DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState);
begin
  TControlCanvas(Canvas).UpdateTextFlags;
  if Assigned(OnDrawItem) then
    OnDrawItem(Self, Index, Rect, State)
  else begin
    Canvas.FillRect(Rect);
    if Index >= 0 then
      WideCanvasTextOut(Canvas, Rect.Left + 2, Rect.Top, Items[Index]);
  end;
end;

function TTntCustomComboBox.GetItems: TTntWideStrings;
begin
  Result := FItems;
end;

procedure TTntCustomComboBox.SetItems(const Value: TTntWideStrings);
begin
  FItems.Assign(Value);
end;

function TTntCustomComboBox.GetSelText: WideString;
begin
  Result := TntCombo_GetSelText(Self);
end;

procedure TTntCustomComboBox.SetSelText(const Value: WideString);
begin
  TntCombo_SetSelText(Self, Value);
end;

function TTntCustomComboBox.GetText: WideString;
begin
  Result := TntControl_GetText(Self);
end;

procedure TTntCustomComboBox.SetText(const Value: WideString);
begin
  TntControl_SetText(Self, Value);
end;

procedure TTntCustomComboBox.CNCommand(var Message: TWMCommand);
begin
  if not TntCombo_CNCommand(Self, Message) then
    inherited;
end;

function TTntCustomComboBox.GetSelLength: Integer;
begin
  Result := SelLength;
end;

procedure TTntCustomComboBox.SetSelLength(Value: Integer);
begin
  SelLength := Value;
end;

function TTntCustomComboBox.GetSelStart: Integer;
begin
  Result := SelStart;
end;

procedure TTntCustomComboBox.SetSelStart(Value: Integer);
begin
  SelStart := Value;
end;

function TTntCustomComboBox.GetItemIndex: Integer;
begin
  Result := ItemIndex;
end;

procedure TTntCustomComboBox.SetItemIndex(const Value: Integer);
begin
  ItemIndex := Value;
end;

function TTntCustomComboBox.GetAutoComplete_PreserveDataEntryCase: Boolean;
begin
  Result := True;
end;

function TTntCustomComboBox.GetAutoComplete_UniqueMatchOnly: Boolean;
begin
  Result := False;
end;

function TTntCustomComboBox.IsHintStored: Boolean;
begin
  Result := TntControl_IsHintStored(Self)
end;

function TTntCustomComboBox.GetHint: WideString;
begin
  Result := TntControl_GetHint(Self)
end;

procedure TTntCustomComboBox.SetHint(const Value: WideString);
begin
  TntControl_SetHint(Self, Value);
end;

procedure TTntCustomComboBox.AddItem(const Item: WideString; AObject: TObject);
begin
  TntComboBox_AddItem(Items, Item, AObject);
end;

{$IFNDEF VER130}
procedure TTntCustomComboBox.CopySelection(Destination: TCustomListControl);
begin
  TntComboBox_CopySelection(Items, ItemIndex, Destination);
end;
{$ENDIF}

procedure TTntCustomComboBox.ActionChange(Sender: TObject; CheckDefaults: Boolean);
begin
  TntControl_BeforeInherited_ActionChange(Self, Sender, CheckDefaults);
  inherited;
end;

function TTntCustomComboBox.GetActionLinkClass: TControlActionLinkClass;
begin
  Result := TntControl_GetActionLinkClass(Self, inherited GetActionLinkClass);
end;

{$IFDEF VER150}
function TTntCustomComboBox.GetItemsClass: TCustomComboBoxStringsClass;
begin
  Result := TD7PatchedComboBoxStrings;
end;
{$ENDIF}

{ TTntListBoxStrings }

function TTntListBoxStrings.GetListBox: TCustomListBox{TNT-ALLOW TCustomListBox};
begin
  Result := TCustomListBox{TNT-ALLOW TCustomListBox}(FListBox);
end;

procedure TTntListBoxStrings.SetListBox(const Value: TCustomListBox{TNT-ALLOW TCustomListBox});
begin
  FListBox := TAccessCustomListBox(Value);
end;

function TTntListBoxStrings.GetCount: Integer;
begin
  Result := ListBox.Items.Count;
end;

function TTntListBoxStrings.Get(Index: Integer): WideString;
var
  Len: Integer;
begin
  if (not IsWindowUnicode(ListBox.Handle)) then
    Result := ListBox.Items[Index]
  else begin
    Len := SendMessageW(ListBox.Handle, LB_GETTEXTLEN, Index, 0);
    if Len = LB_ERR then
      Error(SListIndexError, Index)
    else begin
      SetLength(Result, Len + 1);
      Len := SendMessageW(ListBox.Handle, LB_GETTEXT, Index, Longint(PWideChar(Result)));
      if Len = LB_ERR then
        Result := ''
       else
        Result := PWideChar(Result);
    end;
  end;
end;

function TTntListBoxStrings.GetObject(Index: Integer): TObject;
begin
  Result := ListBox.Items.Objects[Index];
end;

procedure TTntListBoxStrings.Put(Index: Integer; const S: WideString);
var
  I: Integer;
  TempData: Longint;
begin
  I := ListBox.ItemIndex;
  TempData := FListBox.InternalGetItemData(Index);
  // Set the Item to 0 in case it is an object that gets freed during Delete
  FListBox.InternalSetItemData(Index, 0);
  Delete(Index);
  InsertObject(Index, S, nil);
  FListBox.InternalSetItemData(Index, TempData);
  ListBox.ItemIndex := I;
end;

procedure TTntListBoxStrings.PutObject(Index: Integer; AObject: TObject);
begin
  ListBox.Items.Objects[Index] := AObject;
end;

function TTntListBoxStrings.Add(const S: WideString): Integer;
begin
  if (not IsWindowUnicode(ListBox.Handle)) then
    Result := ListBox.Items.Add(S)
  else begin
    Result := SendMessageW(ListBox.Handle, LB_ADDSTRING, 0, Longint(PWideChar(S)));
    if Result < 0 then
      raise EOutOfResources.Create(SInsertLineError);
  end;
end;

procedure TTntListBoxStrings.Insert(Index: Integer; const S: WideString);
begin
  if (not IsWindowUnicode(ListBox.Handle)) then
    ListBox.Items.Insert(Index, S)
  else begin
    if SendMessageW(ListBox.Handle, LB_INSERTSTRING, Index, Longint(PWideChar(S))) < 0 then
      raise EOutOfResources.Create(SInsertLineError);
  end;
end;

procedure TTntListBoxStrings.Delete(Index: Integer);
begin
  FListBox.DeleteString(Index);
end;

procedure TTntListBoxStrings.Exchange(Index1, Index2: Integer);
var
  TempData: Longint;
  TempString: WideString;
begin
  BeginUpdate;
  try
    TempString := Strings[Index1];
    TempData := FListBox.InternalGetItemData(Index1);
    Strings[Index1] := Strings[Index2];
    FListBox.InternalSetItemData(Index1, FListBox.InternalGetItemData(Index2));
    Strings[Index2] := TempString;
    FListBox.InternalSetItemData(Index2, TempData);
    if ListBox.ItemIndex = Index1 then
      ListBox.ItemIndex := Index2
    else if ListBox.ItemIndex = Index2 then
      ListBox.ItemIndex := Index1;
  finally
    EndUpdate;
  end;
end;

procedure TTntListBoxStrings.Clear;
begin
  FListBox.ResetContent;
end;

procedure TTntListBoxStrings.SetUpdateState(Updating: Boolean);
begin
  TAccessStrings(ListBox.Items).SetUpdateState(Updating);
end;

function TTntListBoxStrings.IndexOf(const S: WideString): Integer;
begin
  if (not IsWindowUnicode(ListBox.Handle)) then
    Result := ListBox.Items.IndexOf(S)
  else
    Result := SendMessageW(ListBox.Handle, LB_FINDSTRINGEXACT, -1, LongInt(PWideChar(S)));
end;

procedure TTntListBoxStrings.Move(CurIndex, NewIndex: Integer);
var
  TempData: Longint;
  TempString: WideString;
begin
  BeginUpdate;
  FListBox.FMoving := True;
  try
    if CurIndex <> NewIndex then
    begin
      TempString := Get(CurIndex);
      TempData := FListBox.InternalGetItemData(CurIndex);
      FListBox.InternalSetItemData(CurIndex, 0);
      Delete(CurIndex);
      Insert(NewIndex, TempString);
      FListBox.InternalSetItemData(NewIndex, TempData);
    end;
  finally
    FListBox.FMoving := False;
    EndUpdate;
  end;
end;

//-- list box helper procs

procedure TntListBox_AfterInherited_CreateWnd(ListBox: TCustomListBox{TNT-ALLOW TCustomListBox};
  var FSaveItems: TTntWideStrings; FItems: TTntWideStrings; FSaveTopIndex, FSaveItemIndex: Integer);
begin
  if FSaveItems <> nil then
  begin
    FItems.Assign(FSaveItems);
    FreeAndNil(FSaveItems);
    ListBox.TopIndex := FSaveTopIndex;
    ListBox.ItemIndex := FSaveItemIndex;
  end;
end;

procedure TntListBox_BeforeInherited_DestroyWnd(ListBox: TCustomListBox{TNT-ALLOW TCustomListBox};
  var FSaveItems: TTntWideStrings; const FItems: TTntWideStrings; var FSaveTopIndex, FSaveItemIndex: Integer);
begin
  if (FItems.Count > 0)
{$IFNDEF VER130}
  and (not (TAccessCustomListBox(ListBox).Style in [lbVirtual, lbVirtualOwnerDraw]))
{$ENDIF}
  then begin
    FSaveItems := TTntWideStringList.Create;
    FSaveItems.Assign(FItems);
    FSaveTopIndex := ListBox.TopIndex;
    FSaveItemIndex := ListBox.ItemIndex;
  end;
end;

procedure TntListBox_DrawItem_Text(ListBox: TCustomListBox{TNT-ALLOW TCustomListBox}; Items: TTntWideStrings; Index: Integer; Rect: TRect);
var
  Flags: Integer;
  Canvas: TCanvas;
begin
  Canvas := TAccessCustomListBox(ListBox).Canvas;
  Canvas.FillRect(Rect);
  if Index < Items.Count then
  begin
    Flags := ListBox.DrawTextBiDiModeFlags(DT_SINGLELINE or DT_VCENTER or DT_NOPREFIX);
    if not ListBox.UseRightToLeftAlignment then
      Inc(Rect.Left, 2)
    else
      Dec(Rect.Right, 2);
    Tnt_DrawTextW(Canvas.Handle, PWideChar(Items[Index]), Length(Items[Index]), Rect, Flags);
  end;
end;

procedure TntListBox_AddItem(Items: TTntWideStrings; const Item: WideString; AObject: TObject);
begin
  Items.AddObject(PWideChar(Item), AObject);
end;

{$IFNDEF VER130}
procedure TntListBox_CopySelection(ListBox: TCustomListbox{TNT-ALLOW TCustomListbox};
  Items: TTntWideStrings; Destination: TCustomListControl);
var
  I: Integer;
begin
  if ListBox.MultiSelect then
  begin
    for I := 0 to Items.Count - 1 do
      if ListBox.Selected[I] then
        WideListControl_AddItem(Destination, PWideChar(Items[I]), Items.Objects[I]);
  end
  else
    if Listbox.ItemIndex <> -1 then
      WideListControl_AddItem(Destination, PWideChar(Items[ListBox.ItemIndex]), Items.Objects[ListBox.ItemIndex]);
end;
{$ENDIF}

{ TTntCustomListBox }

constructor TTntCustomListBox.Create(AOwner: TComponent);
begin
  inherited;
  FItems := TTntListBoxStrings.Create;
  TTntListBoxStrings(FItems).ListBox := Self;
end;

destructor TTntCustomListBox.Destroy;
begin
  FreeAndNil(FItems);
  FreeAndNil(FSaveItems);
  inherited;
end;

procedure TTntCustomListBox.CreateWindowHandle(const Params: TCreateParams);
begin
  CreateUnicodeHandle(Self, Params, 'LISTBOX');
end;

procedure TTntCustomListBox.DefineProperties(Filer: TFiler);
begin
  inherited;
  DefineWideProperties(Filer, Self);
end;

procedure TTntCustomListBox.CreateWnd;
begin
  inherited;
  TntListBox_AfterInherited_CreateWnd(Self, FSaveItems, FItems, FSaveTopIndex, FSaveItemIndex);
end;

procedure TTntCustomListBox.DestroyWnd;
begin
  TntListBox_BeforeInherited_DestroyWnd(Self, FSaveItems, FItems, FSaveTopIndex, FSaveItemIndex);
  inherited;
end;

procedure TTntCustomListBox.SetItems(const Value: TTntWideStrings);
begin
  FItems.Assign(Value);
end;

procedure TTntCustomListBox.DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState);
begin
  if Assigned(OnDrawItem) then
    OnDrawItem(Self, Index, Rect, State)
  else
    TntListBox_DrawItem_Text(Self, Items, Index, Rect);
end;

function TTntCustomListBox.IsHintStored: Boolean;
begin
  Result := TntControl_IsHintStored(Self)
end;

function TTntCustomListBox.GetHint: WideString;
begin
  Result := TntControl_GetHint(Self)
end;

procedure TTntCustomListBox.SetHint(const Value: WideString);
begin
  TntControl_SetHint(Self, Value);
end;

procedure TTntCustomListBox.AddItem(const Item: WideString; AObject: TObject);
begin
  TntListBox_AddItem(Items, Item, AObject);
end;

{$IFNDEF VER130}
procedure TTntCustomListBox.CopySelection(Destination: TCustomListControl);
begin
  TntListBox_CopySelection(Self, Items, Destination);
end;
{$ENDIF}

procedure TTntCustomListBox.ActionChange(Sender: TObject; CheckDefaults: Boolean);
begin
  TntControl_BeforeInherited_ActionChange(Self, Sender, CheckDefaults);
  inherited;
end;

function TTntCustomListBox.GetActionLinkClass: TControlActionLinkClass;
begin
  Result := TntControl_GetActionLinkClass(Self, inherited GetActionLinkClass);
end;

{ TTntCustomLabel }

procedure TTntCustomLabel.CMDialogChar(var Message: TCMDialogChar);
begin
  if (FocusControl <> nil) and Enabled and ShowAccelChar and
    IsWideCharAccel(Message.CharCode, Caption) then
    with FocusControl do
      if CanFocus then
      begin
        SetFocus;
        Message.Result := 1;
      end;
end;

function TTntCustomLabel.IsCaptionStored: Boolean;
begin
  Result := TntControl_IsCaptionStored(Self)
end;

function TTntCustomLabel.GetCaption: TWideCaption;
begin
  Result := TntControl_GetText(Self);
end;

procedure TTntCustomLabel.SetCaption(const Value: TWideCaption);
begin
  TntControl_SetText(Self, Value);
end;

procedure TTntCustomLabel.DefineProperties(Filer: TFiler);
begin
  inherited;
  DefineWideProperties(Filer, Self);
end;

function TTntCustomLabel.GetLabelText: WideString;
begin
  Result := Caption;
end;

procedure TTntCustomLabel.DoDrawText(var Rect: TRect; Flags: Integer);
var
  Text: WideString;
begin
  if not Win32PlatformIsUnicode then
    inherited
  else begin
    Text := GetLabelText;
    if (Flags and DT_CALCRECT <> 0) and ((Text = '') or ShowAccelChar and
      (Text[1] = '&') and (Text[2] = #0)) then Text := Text + ' ';
    if not ShowAccelChar then Flags := Flags or DT_NOPREFIX;
    Flags := DrawTextBiDiModeFlags(Flags);
    Canvas.Font := Font;
    if not Enabled then
    begin
      OffsetRect(Rect, 1, 1);
      Canvas.Font.Color := clBtnHighlight;
      DrawTextW(Canvas.Handle, PWideChar(Text), Length(Text), Rect, Flags);
      OffsetRect(Rect, -1, -1);
      Canvas.Font.Color := clBtnShadow;
      DrawTextW(Canvas.Handle, PWideChar(Text), Length(Text), Rect, Flags);
    end
    else
      DrawTextW(Canvas.Handle, PWideChar(Text), Length(Text), Rect, Flags);
  end;
end;

function TTntCustomLabel.IsHintStored: Boolean;
begin
  Result := TntControl_IsHintStored(Self)
end;

function TTntCustomLabel.GetHint: WideString;
begin
  Result := TntControl_GetHint(Self)
end;

procedure TTntCustomLabel.SetHint(const Value: WideString);
begin
  TntControl_SetHint(Self, Value);
end;

procedure TTntCustomLabel.CMHintShow(var Message: TMessage);
begin
  ProcessCMHintShowMsg(Message);
  inherited;
end;

procedure TTntCustomLabel.ActionChange(Sender: TObject; CheckDefaults: Boolean);
begin
  TntControl_BeforeInherited_ActionChange(Self, Sender, CheckDefaults);
  inherited;
end;

function TTntCustomLabel.GetActionLinkClass: TControlActionLinkClass;
begin
  Result := TntControl_GetActionLinkClass(Self, inherited GetActionLinkClass);
end;

{ TTntButton }

procedure TntButton_CMDialogChar(Button: TButton{TNT-ALLOW TButton}; var Message: TCMDialogChar);
begin
  with Message do
    if IsWideCharAccel(Message.CharCode, TntControl_GetText(Button))
    and Button.CanFocus then
    begin
      Button.Click;
      Result := 1;
    end else
      Button.Broadcast(Message);
end;

procedure TTntButton.CreateWindowHandle(const Params: TCreateParams);
begin
  CreateUnicodeHandle(Self, Params, 'BUTTON');
end;

procedure TTntButton.DefineProperties(Filer: TFiler);
begin
  inherited;
  DefineWideProperties(Filer, Self);
end;

procedure TTntButton.CMDialogChar(var Message: TCMDialogChar);
begin
  TntButton_CMDialogChar(Self, Message);
end;

function TTntButton.IsCaptionStored: Boolean;
begin
  Result := TntControl_IsCaptionStored(Self)
end;

function TTntButton.GetCaption: TWideCaption;
begin
  Result := TntControl_GetText(Self)
end;

procedure TTntButton.SetCaption(const Value: TWideCaption);
begin
  TntControl_SetText(Self, Value);
end;

function TTntButton.IsHintStored: Boolean;
begin
  Result := TntControl_IsHintStored(Self)
end;

function TTntButton.GetHint: WideString;
begin
  Result := TntControl_GetHint(Self)
end;

procedure TTntButton.SetHint(const Value: WideString);
begin
  TntControl_SetHint(Self, Value);
end;

procedure TTntButton.ActionChange(Sender: TObject; CheckDefaults: Boolean);
begin
  TntControl_BeforeInherited_ActionChange(Self, Sender, CheckDefaults);
  inherited;
end;

function TTntButton.GetActionLinkClass: TControlActionLinkClass;
begin
  Result := TntControl_GetActionLinkClass(Self, inherited GetActionLinkClass);
end;

{ TTntCustomCheckBox }

procedure TTntCustomCheckBox.CreateWindowHandle(const Params: TCreateParams);
begin
  CreateUnicodeHandle(Self, Params, 'BUTTON');
end;

procedure TTntCustomCheckBox.DefineProperties(Filer: TFiler);
begin
  inherited;
  DefineWideProperties(Filer, Self);
end;

procedure TTntCustomCheckBox.CMDialogChar(var Message: TCMDialogChar);
begin
  with Message do
    if IsWideCharAccel(Message.CharCode, Caption)
    and CanFocus then
    begin
      SetFocus;
      if Focused then Toggle;
      Result := 1;
    end else
      Broadcast(Message);
end;

function TTntCustomCheckBox.IsCaptionStored: Boolean;
begin
  Result := TntControl_IsCaptionStored(Self)
end;

function TTntCustomCheckBox.GetCaption: TWideCaption;
begin
  Result := TntControl_GetText(Self)
end;

procedure TTntCustomCheckBox.SetCaption(const Value: TWideCaption);
begin
  TntControl_SetText(Self, Value);
end;

function TTntCustomCheckBox.IsHintStored: Boolean;
begin
  Result := TntControl_IsHintStored(Self)
end;

function TTntCustomCheckBox.GetHint: WideString;
begin
  Result := TntControl_GetHint(Self)
end;

procedure TTntCustomCheckBox.SetHint(const Value: WideString);
begin
  TntControl_SetHint(Self, Value);
end;

procedure TTntCustomCheckBox.ActionChange(Sender: TObject; CheckDefaults: Boolean);
begin
  TntControl_BeforeInherited_ActionChange(Self, Sender, CheckDefaults);
  inherited;
end;

function TTntCustomCheckBox.GetActionLinkClass: TControlActionLinkClass;
begin
  Result := TntControl_GetActionLinkClass(Self, inherited GetActionLinkClass);
end;

{ TTntRadioButton }

procedure TTntRadioButton.CreateWindowHandle(const Params: TCreateParams);
begin
  CreateUnicodeHandle(Self, Params, 'BUTTON');
end;

procedure TTntRadioButton.DefineProperties(Filer: TFiler);
begin
  inherited;
  DefineWideProperties(Filer, Self);
end;

procedure TTntRadioButton.CMDialogChar(var Message: TCMDialogChar);
begin
  with Message do
    if IsWideCharAccel(Message.CharCode, Caption)
    and CanFocus then
    begin
      SetFocus;
      Result := 1;
    end else
      Broadcast(Message);
end;

function TTntRadioButton.IsCaptionStored: Boolean;
begin
  Result := TntControl_IsCaptionStored(Self);
end;

function TTntRadioButton.GetCaption: TWideCaption;
begin
  Result := TntControl_GetText(Self)
end;

procedure TTntRadioButton.SetCaption(const Value: TWideCaption);
begin
  TntControl_SetText(Self, Value);
end;

function TTntRadioButton.IsHintStored: Boolean;
begin
  Result := TntControl_IsHintStored(Self)
end;

function TTntRadioButton.GetHint: WideString;
begin
  Result := TntControl_GetHint(Self)
end;

procedure TTntRadioButton.SetHint(const Value: WideString);
begin
  TntControl_SetHint(Self, Value);
end;

procedure TTntRadioButton.ActionChange(Sender: TObject; CheckDefaults: Boolean);
begin
  TntControl_BeforeInherited_ActionChange(Self, Sender, CheckDefaults);
  inherited;
end;

function TTntRadioButton.GetActionLinkClass: TControlActionLinkClass;
begin
  Result := TntControl_GetActionLinkClass(Self, inherited GetActionLinkClass);
end;

{ TTntScrollBar }

procedure TTntScrollBar.CreateWindowHandle(const Params: TCreateParams);
begin
  CreateUnicodeHandle(Self, Params, 'SCROLLBAR');
end;

procedure TTntScrollBar.DefineProperties(Filer: TFiler);
begin
  inherited;
  DefineWideProperties(Filer, Self);
end;

function TTntScrollBar.IsHintStored: Boolean;
begin
  Result := TntControl_IsHintStored(Self)
end;

function TTntScrollBar.GetHint: WideString;
begin
  Result := TntControl_GetHint(Self)
end;

procedure TTntScrollBar.SetHint(const Value: WideString);
begin
  TntControl_SetHint(Self, Value);
end;

procedure TTntScrollBar.ActionChange(Sender: TObject; CheckDefaults: Boolean);
begin
  TntControl_BeforeInherited_ActionChange(Self, Sender, CheckDefaults);
  inherited;
end;

function TTntScrollBar.GetActionLinkClass: TControlActionLinkClass;
begin
  Result := TntControl_GetActionLinkClass(Self, inherited GetActionLinkClass);
end;

{ TTntCustomGroupBox }

procedure TTntCustomGroupBox.CreateWindowHandle(const Params: TCreateParams);
begin
  CreateUnicodeHandle(Self, Params, '');
end;

procedure TTntCustomGroupBox.DefineProperties(Filer: TFiler);
begin
  inherited;
  DefineWideProperties(Filer, Self);
end;

procedure TTntCustomGroupBox.CMDialogChar(var Message: TCMDialogChar);
begin
  with Message do
    if IsWideCharAccel(Message.CharCode, Caption)
    and CanFocus then
    begin
      SelectFirst;
      Result := 1;
    end else
      Broadcast(Message);
end;

function TTntCustomGroupBox.IsCaptionStored: Boolean;
begin
  Result := TntControl_IsCaptionStored(Self);
end;

function TTntCustomGroupBox.GetCaption: TWideCaption;
begin
  Result := TntControl_GetText(Self)
end;

procedure TTntCustomGroupBox.SetCaption(const Value: TWideCaption);
begin
  TntControl_SetText(Self, Value);
end;

procedure TTntCustomGroupBox.Paint;

  {$IFDEF VER150}
  procedure PaintThemedGroupBox;
  var
    CaptionRect: TRect;
    OuterRect: TRect;
    Size: TSize;
    Box: TThemedButton;
    Details: TThemedElementDetails;
  begin
    with Canvas do begin
      if Caption <> '' then
      begin
        GetTextExtentPoint32W(Handle, PWideChar(Caption), Length(Caption), Size);
        CaptionRect := Rect(0, 0, Size.cx, Size.cy);
        if not UseRightToLeftAlignment then
          OffsetRect(CaptionRect, 8, 0)
        else
          OffsetRect(CaptionRect, Width - 8 - CaptionRect.Right, 0);
      end
      else
        CaptionRect := Rect(0, 0, 0, 0);

      OuterRect := ClientRect;
      OuterRect.Top := (CaptionRect.Bottom - CaptionRect.Top) div 2;
      with CaptionRect do
        ExcludeClipRect(Handle, Left, Top, Right, Bottom);
      if Enabled then
        Box := tbGroupBoxNormal
      else
        Box := tbGroupBoxDisabled;
      Details := ThemeServices.GetElementDetails(Box);
      ThemeServices.DrawElement(Handle, Details, OuterRect);

      SelectClipRgn(Handle, 0);
      if Text <> '' then
        ThemeServices.DrawText{TNT-ALLOW DrawText}(Handle, Details, Caption, CaptionRect, DT_LEFT, 0);
    end;
  end;
  {$ENDIF}

  procedure PaintGroupBox;
  var
    H: Integer;
    R: TRect;
    Flags: Longint;
  begin
    with Canvas do begin
      H := WideCanvasTextHeight(Canvas, '0');
      R := Rect(0, H div 2 - 1, Width, Height);
      if Ctl3D then
      begin
        Inc(R.Left);
        Inc(R.Top);
        Brush.Color := clBtnHighlight;
        FrameRect(R);
        OffsetRect(R, -1, -1);
        Brush.Color := clBtnShadow;
      end else
        Brush.Color := clWindowFrame;
      FrameRect(R);
      if Caption <> '' then
      begin
        if not UseRightToLeftAlignment then
          R := Rect(8, 0, 0, H)
        else
          R := Rect(R.Right - WideCanvasTextWidth(Canvas, Caption) - 8, 0, 0, H);
        Flags := DrawTextBiDiModeFlags(DT_SINGLELINE);
        DrawTextW(Handle, PWideChar(Caption), Length(Caption), R, Flags or DT_CALCRECT);
        Brush.Color := Color;
        DrawTextW(Handle, PWideChar(Caption), Length(Caption), R, Flags);
      end;
    end;
  end;

begin
  if (not Win32PlatformIsUnicode) then
    inherited
  else
  begin
    Canvas.Font := Self.Font;
    {$IFDEF VER150}
    if ThemeServices.ThemesEnabled then
      PaintThemedGroupBox
    else
      PaintGroupBox;
    {$ELSE}
    PaintGroupBox;
    {$ENDIF}
  end;
end;

function TTntCustomGroupBox.IsHintStored: Boolean;
begin
  Result := TntControl_IsHintStored(Self)
end;

function TTntCustomGroupBox.GetHint: WideString;
begin
  Result := TntControl_GetHint(Self);
end;

procedure TTntCustomGroupBox.SetHint(const Value: WideString);
begin
  TntControl_SetHint(Self, Value);
end;

procedure TTntCustomGroupBox.ActionChange(Sender: TObject; CheckDefaults: Boolean);
begin
  TntControl_BeforeInherited_ActionChange(Self, Sender, CheckDefaults);
  inherited;
end;

function TTntCustomGroupBox.GetActionLinkClass: TControlActionLinkClass;
begin
  Result := TntControl_GetActionLinkClass(Self, inherited GetActionLinkClass);
end;

{ TTntCustomStaticText }

procedure TTntCustomStaticText.CreateWindowHandle(const Params: TCreateParams);
begin
  CreateUnicodeHandle(Self, Params, 'STATIC');
end;

procedure TTntCustomStaticText.DefineProperties(Filer: TFiler);
begin
  inherited;
  DefineWideProperties(Filer, Self);
end;

procedure TTntCustomStaticText.CMDialogChar(var Message: TCMDialogChar);
begin
  with Message do
    if IsWideCharAccel(Message.CharCode, Caption)
      and CanFocus then
    begin
      Click;
      Result := 1;
    end else
      Broadcast(Message);
end;

function TTntCustomStaticText.IsCaptionStored: Boolean;
begin
  Result := TntControl_IsCaptionStored(Self)
end;

function TTntCustomStaticText.GetCaption: TWideCaption;
begin
  Result := TntControl_GetText(Self)
end;

procedure TTntCustomStaticText.SetCaption(const Value: TWideCaption);
begin
  TntControl_SetText(Self, Value);
end;

function TTntCustomStaticText.IsHintStored: Boolean;
begin
  Result := TntControl_IsHintStored(Self)
end;

function TTntCustomStaticText.GetHint: WideString;
begin
  Result := TntControl_GetHint(Self)
end;

procedure TTntCustomStaticText.SetHint(const Value: WideString);
begin
  TntControl_SetHint(Self, Value);
end;

procedure TTntCustomStaticText.ActionChange(Sender: TObject; CheckDefaults: Boolean);
begin
  TntControl_BeforeInherited_ActionChange(Self, Sender, CheckDefaults);
  inherited;
end;

function TTntCustomStaticText.GetActionLinkClass: TControlActionLinkClass;
begin
  Result := TntControl_GetActionLinkClass(Self, inherited GetActionLinkClass);
end;

end.
